# Functions on Null Values {#functions-on-nulls}

Binary functions or operators involving null values return null results.

``` {#codeblock_p5w_1vm_tdc}

x = 1.0 + 5.6 * 3 + NULL + 3
isNull(x)
// output: 1
typestr x;
// output: DOUBLE

5 + NULL      // return null

x=1 2 NULL NULL 3;
y=2 NULL NULL 3 4;
x+y;
// output: [3,,,,7]

false || 00i     // return null
true && 00i      // return null


```

In comparison operators `>`, `>=`, `<`, `<=`, and `between`, null values are treated as the minimum value by default. To have comparisons involving null values return a logical null result, set the configuration parameter *nullAsMinValueForComparison* to false. Nonetheless, in operators `!=`, `<>`, and `==`, null values are always treated as the minimum regardless of this parameter. Therefore, in DolphinDB, comparing two null values for equality returns true, and for inequality returns false.

``` {#codeblock_e5d_cvm_tdc}
// nullAsMinValueForComparison = true
1 < NULL // output: false
1 > NULL // output: true
NULL == NULL // output: true
NULL != NULL // output: false

// nullAsMinValueForComparison = false
1 < NULL // output: 00b
1 > NULL // output: 00b
NULL == NULL // output: true
NULL != NULL // output: false
```

In logical operators `and` and `xor`, null values are treated the same as in basic arithmetic operations: if any operand is null, the result is null.

``` {#codeblock_v5g_fzm_tdc}

false || 00i     // return null
true && 00i      // return null
```

The logical operator `or` processes null values differently. By default \(with *logicOrIgnoreNull*=true\), if only one operand is null, the result matches the other non-null operand. If both operands are null, the result is also null. However, when *logicOrIgnoreNull* is set to false, any null operand leads to a null result.

``` {#codeblock_e5m_yym_tdc}
// logicOrIgnoreNull = true
NULL or true // output: true
NULL or false // output: false
NULL or NULL // output: 00b

// logicOrIgnoreNull = false

NULL or true // output: 00b
NULL or false // output: 00b
NULL or NULL // output: 00b
```

Aggregate functions such as `sum`/`avg`/`med` ignore null values, as do vector functions involving these aggregate functions, such as `msum` and `cumsum`.

``` {#codeblock_mnx_5hn_ryb}
x = 1 2 NULL NULL 3;
log x;
// output: [0,0.693147,,,1.098612]

avg x;
// output: 2

sum x;
// output: 6

med x;
// output: 2

cumsum(x)
// output: 1 3 3 3 6
```

In functions `ols`/`olsEx`/`corrMatrix`/`olsEx`, null values in parameters are replaced with 0.

``` {#codeblock_mwc_k1n_tdc}
m = take(rand(10.0, 20) join NULL, 30) $10:3
corrMatrix(m)

```

|col1|col2|col3|
|----|----|----|
|1|-0.13|-0.0463|
|-0.13|1|0.5853|
|-0.0463|0.5853|1|

**Parent topic:**[Null Value Operations](../../../../Programming/DataManipulation/NullValueManipulation/NullValueOperation/NullValueOperation.md)

