# remoteRun {#remoterun}

## Syntax {#syntax}

`remoteRun(conn, script, args)`

## Arguments {#arguments}

**conn** represents a database connection.

**script** is a string indicating the script to be executed on the remote node.

**args** ... are the parameters for the function to be executed if *script* is a function name. It can have 0 items or multiple items.

## Details {#details}

Send a script or function to a remote database for execution. The `remoteRun` function requires version compatibility when the local server is 3.00 or higher.

## Examples {#examples}

The first use case: execute script on a remote node.

```
conn =  xdb("localhost",81);
remoteRun(conn, "x=rand(1.0,10000000); y=x pow 2; avg y");
// output: 0.333254
```

The second use case:

-   If *functionName* is quoted: execute a remote function on a remote node. The function is defined on the remote node, while the parameters of the function are given on the local node.

    ``` {#codeblock_sbd_zy2_czb .python}
    h=xdb("localhost",80);
    x=remoteRun(h, "sum",1..100);
    x;
    // output: 5050
    ```

-   If *functionName* is not quoted: execute a local function on a remote node. The parameters of the function are given on the local node.


Assume on the local node we have a table "EarningsDates" with 2 columns: ticker and date. This table has the earnings announcement dates of 3 stocks for the 3rd quarter of 2006. There is a files *USPrices.csv* on a remote node with machine name "localhost" and port number 8081. It contains daily stock prices for all US stocks. We would like to get the stock prices from the remote node for all stocks in the table "EarningsDates" for the week after earnings announcement.

On the remote node, we import the data file to create the table "USPrices", and then share it across all nodes as "sharedUSPrices".

```
USPrices = loadText("c:/DolphinDB/Data/USPrices.csv");
share USPrices as USPrices;
```

When we create a connection to a remote node, the remote node will create a new session for this connection. This new session is completely isolated from other sessions on the remote node. This is convenient for developers as they don't have to worry about name conflicts. In this case, however, we do want to share data among multiple sessions on the same node. We can use the statement [share](../../Programming/ProgrammingStatements/share.md) to share the objects. Currently only tables can be shared in DolphinDB.

On the local node, we create the table `EarningsDates`, and send it with the script over to the remote node. After the execution, the result is sent back to the local node.

```
EarningsDates=table(`XOM`AAPL`IBM as TICKER, 2006.10.26 2006.10.19 2006.10.17 as date)
def loadDailyPrice(data){
    dateDict = dict(data.TICKER, data.date)
    return select date, TICKER, PRC from objByName("sharedUSPrices") where dateDict[TICKER]<date<=dateDict[TICKER]+7
}
conn = xdb("localhost",8081)
prices = conn.remoteRun(loadDailyPrice, EarningsDates);
prices;
```

|date|TICKER|PRC|
|----|------|---|
|2006.10.27|XOM|71.46|
|2006.10.30|XOM|70.84|
|2006.10.31|XOM|71.42|
|2006.11.01|XOM|71.06|
|2006.11.02|XOM|71.19|
|2006.10.18|IBM|89.82|
|2006.10.19|IBM|89.86|
|2006.10.20|IBM|90.48|
|2006.10.23|IBM|91.56|
|2006.10.24|IBM|91.49|
|2006.10.20|AAPL|79.95|
|2006.10.23|AAPL|81.46|
|2006.10.24|AAPL|81.05|
|2006.10.25|AAPL|81.68|
|2006.10.26|AAPL|82.19|

