# pchipInterpolateFit {#pchipInterpolateFit}

**Parent topic:**[Functions](../../Functions/category.md)

## Syntax {#syntax}

`pchipInterpolateFit(X, Y, [extrapolate=true])`

## Arguments {#arguments}

**X** is a numeric vector representing the x-coordinates \(independent variable\) of the interpolation points. *X* must be strictly increasing and contain at least two elements.

**Y** is a numeric vector of the same length as *X*, representing the y-coordinates \(dependent variable\) of the interpolation points.

**extrapolate** \(optional\) is a Boolean scalar indicating whether to allow extrapolation when prediction points fall outside the data range. Defaults to true.

## Details {#details}

This function applies piecewise cubic Hermite interpolation to the given numerical vectors X and Y. PCHIP stands for Piecewise Cubic Hermite Interpolating Polynomial.

**Return value:** A dictionary with the following keys:

-   modelName: A string "PchipInterpolate" indicating the model name.
-   X: The DOUBLE vector representing the x-coordinates used for interpolation \(i.e., the input *X*\).
-   extrapolate: A Boolean scalar. The input *extrapolate*.
-   coeffs: A numeric vector containing the polynomial coefficients fitted from the input data points.
-   predict: The prediction function of the model. You can call `model.predict(X)` or `predict(model, X)` to make predictions with the generated model.
    -   model: The output dictionary of `pchipInterpolateFit`.
    -   X: A numeric vector representing the x-coordinates.

## Examples {#examples}

Perform piecewise cubic Hermite interpolation on the given vectors *x* and *y*.

``` {#codeblock_zq1_tms_c2c}
def linspace(start, end, num, endpoint=true){
	if(endpoint) return end$DOUBLE\(num-1), start + end$DOUBLE\(num-1)*0..(num-1)
	else return start + end$DOUBLE\(num-1)*0..(num-1)	
}

x_observed = linspace(0.0, 10.0, 11)[1]
y_observed = sin(x_observed)

model = pchipInterpolateFit(x_observed, y_observed)
model;

/* output:
modelName->PchipInterpolate
X->[0,1,2,3,4,5,6,7,8,9,10]
coeffs->#0                 #1                 #2                 #3                
------------------ ------------------ ------------------ ------------------
-0.32911446845195  -0.057707802943105 1.228293256202952  0                 
-0.01011863907468  -0.047589163868426 0.125534244960891  0.841470984807897 
0.708356730424705  -1.476534149190519 0                  0.909297426825682 
0.637878921149598  -0.707803317410469 -0.827998107106924 0.141120008059867 
0.074275580231352  0.053570618892506  -0.329967978479069 -0.756802495307928
-0.571482233207003 1.250991009671215  0                  -0.958924274663138
-0.594663658922633 0.743530436118926  0.787535319721422  -0.279415498198926
-0.174138080617811 0.015904513331029  0.490605215191374  0.656986598718789 
0.434603140426252  -1.011842901807877 0                  0.989358246623382 
0.046813296419378  -0.283076510213506 -0.719876382336998 0.412118485241757 

extrapolate->true
predict->cubicHermiteSplinePredict
*/
```

**Related**: [predict](predict.md) [cubicHermiteSplineFit](../c/cubichermitesplinefit.md)

