# linearInterpolateFit {#linearInterpolateFit}

**Parent topic:**[Functions](../../Functions/category.md)

## Syntax {#syntax}

`linearInterpolateFit(X, Y, [fillValue], [sorted=false])`

## Arguments {#arguments}

**X** is a numeric vector indicating the x-coordinates of the points for interpolation. Note that *X* must contain no less than two unique values with no null values.

**Y** is a numeric vector indicating the y-coordinates of the points for interpolation. Note that *Y* must be of the same length as *X* with no null values.

**fillValue** \(optional\) specifies how to assign values for the x-coordinate of the points outside the existing data range. The following options are supported:

-   A numeric pair in the form `(min, max)`, where `min` and `max` represent the values assigned when the x-coordinate of the point Xnew is smaller than the minimum of *X* or larger than the maximum of *X*, respectively. Specifically:
    -   If Xnew &lt; Xmin, it is assigned `below`.
    -   If Xnew &gt; Xmax, it is assigned `above`.
-   The string "extrapolate" \(default\), which indicates that extrapolation is performed.

**sorted**\(optional\) is a Boolean scalar indicating whether the input *X* is sorted in ascending order.

-   If set to true, *X* must be in ascending order.
-   If set to false \(default\), the function will sort *X* and adjust the order of *Y* accordingly.

## Details {#details}

Perform linear interpolation/extrapolation on a set of points. Interpolation estimates unknown values that fall between known data points, while extrapolation estimates values beyond the existing data range.

**Return value**: A dictionary containing the following keys:

-   modelName: A string indicating the model name, which is "linearInterpolate".
-   sortedX: A DOUBLE vector indicating the input *X*sorted in ascending order.
-   sortedY: A DOUBLE vector indicating the input *Y* sorted corresponding to *sortedX*.
-   fillValue: The input *fillValue*.
-   predict: The prediction function of the model, which returns linear interpolation results. It can be called using `model.predict(X)` or `predict(model, X)`, where:
    -   model: A dictionary indicating the output of `linearInterpolateFit`.
    -   X: A numeric vector indicating the x-coordinates of the points to be predicted.

## Examples {#examples}

``` {#codeblock_c2r_qw1_ddc}
def linspace(start, end, num, endpoint=true){
	if(endpoint) return end$DOUBLE\(num-1), start + end$DOUBLE\(num-1)*0..(num-1)
	else return start + (end-start)$DOUBLE\(num)*0..(num-1)	
}
x = 0..9
y = exp(-x/3.0)
model = linearInterpolateFit(x, y, sorted=true)

/*Output
sortedX->[0.0,1.000000000000,2.000000000000,3.000000000000,4.000000000000,5.000000000000,6.000000000000,7.000000000000,8.000000000000,9.000000000000]
modelName->linearInterpolate
predict->linearInterpolatePredict
fillValue->extrapolate
sortedY->[1.000000000000,0.716531310573,0.513417119032,0.367879441171,0.263597138115,0.188875602837,0.135335283236,0.096971967864,0.069483451222,0.049787068367]
*/

// Enter new values of X to predict the corresponding Y values
model.predict(xnew)

//Output：[1,0.829918786344274,0.67590847226555,0.554039957340832,0.455202047888132,0.367879441171442,0.305310059338013,0.248652831060094,0.203819909893195,0.167459474997182,0.135335283236613,0.112317294013288,0.091474264536084,0.074981154551122,0.061604898080826]
```

