# irCrossCurrencyCurveBuilder {#irCrossCurrencyCurveBuilder}

**Parent topic:**[Functions](../../Functions/category.md)

## Syntax {#syntax}

`irCrossCurrencyCurveBuilder(referenceDate, currency, instNames, instTypes, terms, quotes, currencyPair, spot, dayCountConvention, discountCurve, [compounding='Continuous'], [frequency='Annual'], [curveName])`

## Details {#topic_lfr_pxq_ngc}

Builds a cross-currency interest rate swap yield curve.

Return value: A MKTDATA object.

## Arguments {#arguments}

**referenceDate** A DATE scalar specifying the reference date of the yield curve.

**currency** A STRING scalar specifying the currency code of the curve. Supported values: "CNY", "USD", "EUR", "GBP", "JPY", "HKD".

**instNames** A STRING vector indicating the instrument names.

**instTypes** A STRING vector indicating the instrument types.

Currently *instTypes* only supports "FxSwap", and *instNames* should be currency pairs, e.g., "USDCNY".

**terms** A vector of STRING or DURATION type, indicating the remaining maturity, e.g., "1M".

**quotes** A numeric vector indicating the market quotes.

**currencyPair**A STRING scalar specifying the currency pair, in the format "EURUSD", "EUR.USD", or "EUR/USD". Supported currency pairs include:

-   "EURUSD": Euro to US Dollar

-   "USDCNY": US Dollar to Chinese Yuan

-   "EURCNY": Euro to Chinese Yuan

-   "GBPCNY": British Pound to Chinese Yuan

-   "JPYCNY": Japanese Yen to Chinese Yuan

-   "HKDCNY": Hong Kong Dollar to Chinese Yuan


**spot** A DOUBLE scalar specifying the spot exchange rate on the reference date.

**dayCountConvention**is a STRING scalar or vector indicating the day count convention to use. It can be:

-   "ActualActual": actual/actual

-   "Actual360": actual/360

-   "Actual365": actual/365

-   "ActualActualISMA": actual/actual according to ISMA \(International Securities Market Association\) convention

-   "ActualActualISDA": actual/actual according to ISDA \(International Swaps and Derivatives Association\) convention.

-   "Thirty360EU": European 30/360

-   "Thirty360US": US \(NASD\) 30/360


**discountCurve**A MKTDATA object of type IrYieldCurve, indicating discount curve for the other currency in the currency pair, which must be a yield curve built using the bootstrap method. See [Curve Field Specifications](#topic_n4h_sxq_ngc) for details.

**compounding** \(optional\) A STRING scalar defining the compounding method. Options:

-   "Compounded": Discrete compounding

-   "Simple": Simple interest

-   "Continuous" \(default\): Continuous compounding


**frequency** \(optional\) A STRING scalar specifying the interest payment frequency. Supported values:

-   "NoFrequency": No payment frequency

-   "Annual": Annually

-   "Semiannual": Semiannually

-   "EveryFourthMonth": Every four months

-   "Quarterly": Quarterly

-   "BiMonthly": Every two months

-   "Monthly": Monthly

-   "EveryFourthWeek": Every four weeks

-   "BiWeekly": Every two weeks

-   "Weekly": Weekly

-   "Daily": Daily

-   "Other": Other frequencies


**curveName** \(optional\) A STRING scalar indicating the yield curve name. The default value is NULL.

## Examples {#examples}

``` {#codeblock_y55_32x_mgc}
refDate = 2025.08.18
spotDate1 = temporalAdd(refDate, 2, "XNYS")  
spotDate2 = temporalAdd(refDate, 2, "CFET")  
spotDate = max(spotDate1, spotDate2)
instNames = take("USDCNY", 13)
instTypes = take("FxSwap", 13)
terms = ["1d", "1w", "2w", "3w", "1M", "2M", "3M", "6M", "9M", "1y", "18M", "2y", "3y"]
curveDates = array(DATE)
for(term in terms){
    dur = duration(term)
    days1 = transFreq(temporalAdd(spotDate, dur), "XNYS", "right", "right")
    days2 = transFreq(temporalAdd(spotDate, dur), "CFET", "right", "right")
    curveDates.append!(max(transFreq(temporalAdd(spotDate, dur), "XNYS", "right", "right"), transFreq(temporalAdd(spotDate, dur), "CFET", "right", "right")))
 }
quotes = [-5.54, 
          -39.00, 
          -75.40, 
          -113.20, 
          -177.00, 
          -317.00, 
          -466.00, 
          -898.50, 
          -1284.99, 
          -1676.00, 
          -2320.00, 
          -2870.00, 
          -3962.50] \ 10000  // fx swap points

cnyShibor3m = {
    "mktDataType": "Curve",
    "curveType": "IrYieldCurve",
    "version": 0,
    "referenceDate": refDate,
    "currency": "CNY",
    "dayCountConvention": "Actual365",
    "compounding": "Continuous",
    "interpMethod": "Linear",
    "extrapMethod": "Flat",
    "dates": curveDates,
    "values": [1.5113,
               1.5402,
               1.5660,
               1.5574,
               1.5556,
               1.5655,
               1.5703,
               1.5934,
               1.6040,
               1.6020,
               1.5928,
               1.5842,
               1.6068] \ 100,
    "settlement": spotDate
}

cnyShibor3m = parseMktData(cnyShibor3m)

spot = 7.1627
curve = irCrossCurrencyCurveBuilder(refDate, "USD", instNames, instTypes, terms, quotes, "USDCNY", spot, "Actual365", cnyShibor3m, "Continuous")
curveDict = extractMktData(curve)
for(i in 0..(size(quotes)-1) ){
    print(curveDict["values"][i]*100)
}
```

**Releated functions:**[bondYieldCurveBuilder](../b/bondYieldCurveBuilder.md), [irSingleCurrencyCurveBuilder](irSingleCurrencyCurveBuilder.md), [parseMktData](../p/parseMktData.md)

## Curve Field Specifications {#topic_n4h_sxq_ngc}

<table id="table_ok5_3cr_ngc"><thead><tr><th>

Field Name

</th><th>

Data Type

</th><th>

Description

</th><th>

Required

</th></tr></thead><tbody><tr><td>

mktDataType

</td><td>

STRING

</td><td>

Must be "Curve"

</td><td>

√

</td></tr><tr><td>

referenceDate

</td><td>

DATE

</td><td>

Reference Date

</td><td>

√

</td></tr><tr><td>

version

</td><td>

INT

</td><td>

Version, default 0

</td><td>

×

</td></tr><tr><td>

curveType

</td><td>

STRING

</td><td>

Must be "IrYieldCurve"

</td><td>

√

</td></tr><tr><td>

dayCountConvention

</td><td>

STRING

</td><td>

The day count convention to use. It can be:

 -   "Actual360": actual/360

-   "Actual365": actual/365

-   "ActualActualISMA": actual/actual according to ISMA \(International Securities Market Association\) convention

-   "ActualActualISDA": actual/actual according to ISDA \(International Swaps and Derivatives Association\) convention.


</td><td>

√

</td></tr><tr><td>

interpMethod

</td><td>

STRING

</td><td>

Interpolation method. It can be:

 -   "Linear": linear interpolation

-   "CubicSpline": cubic spline interpolation

-   "CubicHermiteSpline": cubic Hermite interpolation


</td><td>

√

</td></tr><tr><td>

extrapMethod

</td><td>

STRING

</td><td>

Extrapolation method. It can be

 -   Flat: flat extrapolation

-   Linear: linear extrapolation


</td><td>

√

</td></tr><tr><td>

dates

</td><td>

DATE vector

</td><td>

Date of each data point

</td><td>

√

</td></tr><tr><td>

values

</td><td>

DOUBLE vector

</td><td>

Value of each data point, corresponding to the elements in dates.

</td><td>

√

</td></tr><tr><td>

curveName

</td><td>

STRING

</td><td>

Curve name

</td><td>

×

</td></tr><tr><td>

currency

</td><td>

STRING

</td><td>

Currency. It can be CNY", "USD", "EUR", "GBP", "JPY", "HKD"

</td><td>

√

</td></tr><tr><td>

compounding

</td><td>

STRING

</td><td>

The compounding interest. It can be:

 -   "Compounded": discrete compounding

-   "Simple": simple interest \(no compounding\).

-   "Continuous": continuous compounding.


</td><td>

√

</td></tr><tr><td>

settlement

</td><td>

DATE

</td><td>

Settlement date. If specified, all subsequent tenor intervals are computed starting from "settlement" rather than from "referenceDate".

</td><td>

×

</td></tr><tr><td>

frequency

</td><td>

INTEGRAL/STRING

</td><td>

The interest payment frequency. Supported values:

 -   -1 or "NoFrequency": No payment frequency

-   0 or "Once": Single lump-sum payment of principal and interest at maturity.

-   1 or "Annual": Annually

-   2 or "Semiannual": Semiannually

-   3 or "EveryFourthMonth": Every four months

-   4 or "Quarterly": Quarterly

-   6 or "BiMonthly": Every two months

-   12 or "Monthly": Monthly

-   13 or "EveryFourthWeek": Every four weeks

-   26 or "BiWeekly": Every two weeks

-   52 or "Weekly": Weekly

-   365 or "Daily": Daily

-   999 or "Other": Other frequencies


</td><td>

×

</td></tr><tr><td>

curveModel

</td><td>

STRING

</td><td>

Curve construction model; Currently, only "Bootstrap" is supported.

</td><td>

×

</td></tr><tr><td>

curveParams

</td><td>

DICT

</td><td>

Model parameters.

</td><td>

×

</td></tr></tbody>
</table>