# fxVolatilitySurfaceBuilder {#fxVolatilitySurfaceBuilder}

**Parent topic:**[Functions](../../Functions/category.md)

## Syntax {#syntax}

`fxVolatilitySurfaceBuilder(referenceDate, currencyPair, quoteNames, quoteTerms, quotes, spot, domesticCurve, foreignCurve, [model='SVI'])`

## Details {#topic_wgm_czq_ngc}

Builds a foreign exchange volatility surface.

**Return value**: A FxVolatilitySurface object of MKTDATA type.

## Arguments {#arguments}

**referenceDate**A DATE scalar specifying the reference date of the surface.

**currencyPair**A STRING scalar specifying the currency pair, in the format "EURUSD", "EUR.USD", or "EUR/USD". Supported currency pairs include:

-   "EURUSD": Euro to US Dollar

-   "USDCNY": US Dollar to Chinese Yuan

-   "EURCNY": Euro to Chinese Yuan

-   "GBPCNY": British Pound to Chinese Yuan

-   "JPYCNY": Japanese Yen to Chinese Yuan

-   "HKDCNY": Hong Kong Dollar to Chinese Yuan


**quoteNames** A STRING vector specifying the names of market volatility quotes，which must be a permutation of \["ATM", "D25\_RR", "D25\_BF", "D10\_RR", "D10\_BF"\], where:

-   "ATM": At-the-money volatility

-   "D25\_RR": Risk reversal with Delta = 0.25

-   "D25\_BF": Butterfly with Delta = 0.25

-   "D10\_RR": Risk reversal with Delta = 0.1

-   "D10\_BF": Butterfly with Delta = 0.1


**quoteTerms** A vector of DURATION or STRING type, representing the tenors associated with market quotes. When it is a STRING scalar, in addition to strings that can be converted into DURATION, the following optional values are also supported:

-   "ON": Overnight, near leg value date is T, far leg value date is T+1

-   "TN": Tomorrow-next, near leg value date is T+1, far leg value date is T+2

-   "SN": Spot-next, near leg value date is T+2, far leg value date is T+3


**quotes** A DOUBLE matrix with shape `(size(quoteTerms), size(quoteNames))`. The entry at row i and column j represents the quote of quoteNames\[j\] for tenor quoteTerms\[i\].

**spot** A DOUBLE scalar representing the spot exchange rate.

**domesticCurve** A MKTDATA object of type IrYieldCurve representing the domestic discount curve. See [Curve Field Specifications](#topic_hhh_fzq_ngc) for details.

**foreignCurve** A MKTDATA object of type IrYieldCurve representing the foreign discount curve. See [Curve Field Specifications](#topic_hhh_fzq_ngc) for details.

**model** \(optional\) A STRING scalar specifying the model used to construct the surface. Options:

-   “SVI” \(default\): Stochastic Volatility Inspired model

-   “SABR”: Stochastic Alpha Beta Rho model

-   “Linear”: Linear interpolation model

-   “CubicSpline”: Cubic spline interpolation model


## Examples {#examples}

``` {#codeblock_y55_32x_mgc}
refDate = 2025.08.18
ccyPair = "USDCNY"
quoteTerms = ['1d', '1w', '2w', '3w', '1M', '2M', '3M', '6M', '9M', '1y', '18M', '2y', '3y']
quoteNames = ["ATM", "D25_RR", "D25_BF", "D10_RR", "D10_BF"]
quotes = [0.030000, -0.007500, 0.003500, -0.010000, 0.005500, 
          0.020833, -0.004500, 0.002000, -0.006000, 0.003800, 
          0.022000, -0.003500, 0.002000, -0.004500, 0.004100, 
          0.022350, -0.003500, 0.002000, -0.004500, 0.004150, 
          0.024178, -0.003000, 0.002200, -0.004750, 0.005500, 
          0.027484, -0.002650, 0.002220, -0.004000, 0.005650, 
          0.030479, -0.002500, 0.002400, -0.003500, 0.005750, 
          0.035752, -0.000500, 0.002750,  0.000000, 0.006950, 
          0.038108,  0.001000, 0.002800,  0.003000, 0.007550, 
          0.039492,  0.002250, 0.002950,  0.005000, 0.007550, 
          0.040500,  0.004000, 0.003100,  0.007000, 0.007850, 
          0.041750,  0.005250, 0.003350,  0.008000, 0.008400, 
          0.044750,  0.006250, 0.003400,  0.009000, 0.008550]
quotes = reshape(quotes, size(quoteNames):size(quoteTerms)).transpose()
spot = 7.1627
curveDates = [2025.08.21,
              2025.08.27,
              2025.09.03,
              2025.09.10,
              2025.09.22,
              2025.10.20,
              2025.11.20,
              2026.02.24,
              2026.05.20,
              2026.08.20,
              2027.02.22,
              2027.08.20,
              2028.08.21]
domesticCurveInfo = {
    "mktDataType": "Curve",
    "curveType": "IrYieldCurve",
    "referenceDate": refDate,
    "currency": "CNY",
    "dayCountConvention": "Actual365",
    "compounding": "Continuous",  
    "interpMethod": "Linear",
    "extrapMethod": "Flat",
    "frequency": "Annual",
    "dates": curveDates,
    "values":[1.5113, 
              1.5402, 
              1.5660, 
              1.5574, 
              1.5556, 
              1.5655, 
              1.5703, 
              1.5934, 
              1.6040, 
              1.6020, 
              1.5928, 
              1.5842, 
              1.6068]/100
}
foreignCurveInfo = {
    "mktDataType": "Curve",
    "curveType": "IrYieldCurve",
    "referenceDate": refDate,
    "currency": "USD",
    "dayCountConvention": "Actual365",
    "compounding": "Continuous",  
    "interpMethod": "Linear",
    "extrapMethod": "Flat",
    "frequency": "Annual",
    "dates": curveDates,
    "values":[4.3345, 
              4.3801, 
              4.3119, 
              4.3065, 
              4.2922, 
              4.2196, 
              4.1599, 
              4.0443, 
              4.0244, 
              3.9698, 
              3.7740, 
              3.6289, 
              3.5003]/100
}
domesticCurve = parseMktData(domesticCurveInfo)
foreignCurve = parseMktData(foreignCurveInfo)
surf = fxVolatilitySurfaceBuilder(refDate, ccyPair, quoteNames, quoteTerms, quotes, spot, domesticCurve, foreignCurve)
surfDict = extractMktData(surf)
print(surfDict)

```

**Related functions:** [extractMktData](../e/extractMktData.md), [parseMktData](../p/parseMktData.md)

## Curve Field Specifications {#topic_hhh_fzq_ngc}

<table id="table_bwd_jn1_4gc"><thead><tr><th>

Field Name

</th><th>

Data Type

</th><th>

Description

</th><th>

Required

</th></tr></thead><tbody><tr><td>

mktDataType

</td><td>

STRING

</td><td>

Must be "Curve"

</td><td>

√

</td></tr><tr><td>

referenceDate

</td><td>

DATE

</td><td>

Reference Date

</td><td>

√

</td></tr><tr><td>

version

</td><td>

INT

</td><td>

Version, default 0

</td><td>

×

</td></tr><tr><td>

curveType

</td><td>

STRING

</td><td>

Must be "IrYieldCurve"

</td><td>

√

</td></tr><tr><td>

dayCountConvention

</td><td>

STRING

</td><td>

The day count convention to use. It can be:

 -   "Actual360": actual/360

-   "Actual365": actual/365

-   "ActualActualISMA": actual/actual according to ISMA \(International Securities Market Association\) convention

-   "ActualActualISDA": actual/actual according to ISDA \(International Swaps and Derivatives Association\) convention.


</td><td>

√

</td></tr><tr><td>

interpMethod

</td><td>

STRING

</td><td>

Interpolation method. It can be:

 -   "Linear": linear interpolation

-   "CubicSpline": cubic spline interpolation

-   "CubicHermiteSpline": cubic Hermite interpolation


</td><td>

√

</td></tr><tr><td>

extrapMethod

</td><td>

STRING

</td><td>

Extrapolation method. It can be

 -   Flat: flat extrapolation

-   Linear: linear extrapolation


</td><td>

√

</td></tr><tr><td>

dates

</td><td>

DATE vector

</td><td>

Date of each data point

</td><td>

√

</td></tr><tr><td>

values

</td><td>

DOUBLE vector

</td><td>

Value of each data point, corresponding to the elements in dates.

</td><td>

√

</td></tr><tr><td>

curveName

</td><td>

STRING

</td><td>

Curve name

</td><td>

×

</td></tr><tr><td>

currency

</td><td>

STRING

</td><td>

Currency. It can be CNY", "USD", "EUR", "GBP", "JPY", "HKD"

</td><td>

√

</td></tr><tr><td>

compounding

</td><td>

STRING

</td><td>

The compounding interest. It can be:

 -   "Compounded": discrete compounding

-   "Simple": simple interest \(no compounding\).

-   "Continuous": continuous compounding.


</td><td>

√

</td></tr><tr><td>

settlement

</td><td>

DATE

</td><td>

Settlement date. If specified, all subsequent tenor intervals are computed starting from "settlement" rather than from "referenceDate".

</td><td>

×

</td></tr><tr><td>

frequency

</td><td>

INTEGRAL/STRING

</td><td>

The interest payment frequency. Supported values:

 -   -1 or "NoFrequency": No payment frequency

-   0 or "Once": Single lump-sum payment of principal and interest at maturity.

-   1 or "Annual": Annually

-   2 or "Semiannual": Semiannually

-   3 or "EveryFourthMonth": Every four months

-   4 or "Quarterly": Quarterly

-   6 or "BiMonthly": Every two months

-   12 or "Monthly": Monthly

-   13 or "EveryFourthWeek": Every four weeks

-   26 or "BiWeekly": Every two weeks

-   52 or "Weekly": Weekly

-   365 or "Daily": Daily

-   999 or "Other": Other frequencies


</td><td>

×

</td></tr><tr><td>

curveModel

</td><td>

STRING

</td><td>

Curve construction model; Currently, only "Bootstrap" is supported.

</td><td>

×

</td></tr><tr><td>

curveParams

</td><td>

DICT

</td><td>

Model parameters.

</td><td>

×

</td></tr></tbody>
</table>