# fxEuropeanOptionPricer {#fxEuropeanOptionPricer}

**Parent topic:**[Functions](../../Functions/category.md)

## Syntax {#syntax}

`fxEuropeanOptionPricer(instrument, pricingDate, spot, domesticCurve, foreignCurve, volSurf)`

## Details {#topic_fyn_2tq_ngc}

Prices a foreign exchange European option and returns its net present value \(NPV\).

**Return value**: A DOUBLE scalar.

## Arguments {#arguments}

**instrument** An INSTRUMENT scalar representing the FX European option to be priced. See [Product Field Specifications](#topic_yzx_3tq_ngc) for details.

**pricingDate** A DATE scalar specifying the valuation date.

**spot**A numeric scalar representing the spot FX rate.

**domesticCurve** A MKTDATA scalar representing the domestic discount curve. See [Curve Field Specifications](#topic_pgr_jtq_ngc) for details.

**foreignCurve** A MKTDATA scalar representing the foreign discount curve. See [Curve Field Specifications](#topic_pgr_jtq_ngc) for details.

**volSurf** A MKTDATA scalar representing the FX volatility surface. See [Curve Field Specifications](#topic_pgr_jtq_ngc) for details.

## Examples {#examples}

``` {#codeblock_y55_32x_mgc}
pricingDate = 2025.08.18
ccyPair = "USDCNY"

option = {
    "productType": "Option",
    "optionType": "EuropeanOption",
    "assetType": "FxEuropeanOption",
    "version": 0,
    "notional": ["USD", 1E6],
    "strike": 7.2,
    "maturity": 2025.10.28,
    "payoffType": "Call",
    "dayCountConvention": "Actual365",
    "underlying": ccyPair
}

quoteTerms = ['1d', '1w', '2w', '3w', '1M', '2M', '3M', '6M', '9M', '1y', '18M', '2y', '3y']
quoteNames = ["ATM", "D25_RR", "D25_BF", "D10_RR", "D10_BF"]
quotes = [0.030000, -0.007500, 0.003500, -0.010000, 0.005500, 
          0.020833, -0.004500, 0.002000, -0.006000, 0.003800, 
          0.022000, -0.003500, 0.002000, -0.004500, 0.004100, 
          0.022350, -0.003500, 0.002000, -0.004500, 0.004150, 
          0.024178, -0.003000, 0.002200, -0.004750, 0.005500, 
          0.027484, -0.002650, 0.002220, -0.004000, 0.005650, 
          0.030479, -0.002500, 0.002400, -0.003500, 0.005750, 
          0.035752, -0.000500, 0.002750,  0.000000, 0.006950, 
          0.038108,  0.001000, 0.002800,  0.003000, 0.007550, 
          0.039492,  0.002250, 0.002950,  0.005000, 0.007550, 
          0.040500,  0.004000, 0.003100,  0.007000, 0.007850, 
          0.041750,  0.005250, 0.003350,  0.008000, 0.008400, 
          0.044750,  0.006250, 0.003400,  0.009000, 0.008550]
quotes = reshape(quotes, size(quoteNames):size(quoteTerms)).transpose()

curveDates = [2025.08.21,
              2025.08.27,
              2025.09.03,
              2025.09.10,
              2025.09.22,
              2025.10.20,
              2025.11.20,
              2026.02.24,
              2026.05.20,
              2026.08.20,
              2027.02.22,
              2027.08.20,
              2028.08.21]
              
domesticCurveInfo = {
    "mktDataType": "Curve",
    "curveType": "IrYieldCurve",
    "referenceDate": pricingDate,
    "currency": "CNY",
    "dayCountConvention": "Actual365",
    "compounding": "Continuous",  
    "interpMethod": "Linear",
    "extrapMethod": "Flat",
    "frequency": "Annual",
    "dates": curveDates,
    "values":[1.5113, 
              1.5402, 
              1.5660, 
              1.5574, 
              1.5556, 
              1.5655, 
              1.5703, 
              1.5934, 
              1.6040, 
              1.6020, 
              1.5928, 
              1.5842, 
              1.6068]/100
}
foreignCurveInfo = {
    "mktDataType": "Curve",
    "curveType": "IrYieldCurve",
    "referenceDate": pricingDate,
    "currency": "USD",
    "dayCountConvention": "Actual365",
    "compounding": "Continuous",  
    "interpMethod": "Linear",
    "extrapMethod": "Flat",
    "frequency": "Annual",
    "dates": curveDates,
    "values":[4.3345, 
              4.3801, 
              4.3119, 
              4.3065, 
              4.2922, 
              4.2196, 
              4.1599, 
              4.0443, 
              4.0244, 
              3.9698, 
              3.7740, 
              3.6289, 
              3.5003]/100
}

spot = 7.1627
instrument = parseInstrument(option)
domesticCurve = parseMktData(domesticCurveInfo)
foreignCurve = parseMktData(foreignCurveInfo)
surf = fxVolatilitySurfaceBuilder(pricingDate, ccyPair, quoteNames, quoteTerms, quotes, spot, domesticCurve, foreignCurve)

npv = fxEuropeanOptionPricer(instrument, pricingDate, spot, domesticCurve, foreignCurve, surf)
print(npv)
```

**Related functions:** [parseInstrument](../p/parseInstrument.md), [parseMktData](../p/parseMktData.md)

## Product Field Specifications {#topic_yzx_3tq_ngc}

<table id="table_pvt_wtq_ngc"><thead><tr><th>

Field Name

</th><th>

Data Type

</th><th>

Description

</th><th>

Required

</th></tr></thead><tbody><tr><td>

productType

</td><td>

STRING

</td><td>

Must be "Option"

</td><td>

√

</td></tr><tr><td>

optionType

</td><td>

STRING

</td><td>

Must be "EuropeanOpiton"

</td><td>

√

</td></tr><tr><td>

assetType

</td><td>

STRING

</td><td>

Must be "FxEuropeanOpiton"

</td><td>

√

</td></tr><tr><td>

version

</td><td>

INT

</td><td>

Version, default 0

</td><td>

×

</td></tr><tr><td>

notional

</td><td>

ANY vector

</td><td>

Notional principal amount, e.g., \["USD", 1.0\]

</td><td>

√

</td></tr><tr><td>

instrumentId

</td><td>

STRING

</td><td>

InstrumentId ID

</td><td>

×

</td></tr><tr><td>

maturity

</td><td>

DATE

</td><td>

Maturity date

</td><td>

√

</td></tr><tr><td>

underlying

</td><td>

STRING

</td><td>

The currency pair, in the format "EURUSD", "EUR.USD", or "EUR/USD". Supported currency pairs include:

 -   "EURUSD": Euro to US Dollar

-   "USDCNY": US Dollar to Chinese Yuan

-   "EURCNY": Euro to Chinese Yuan

-   "GBPCNY": British Pound to Chinese Yuan

-   "JPYCNY": Japanese Yen to Chinese Yuan

-   "HKDCNY": Hong Kong Dollar to Chinese Yuan


</td><td>

√

</td></tr><tr><td>

direction

</td><td>

STRING

</td><td>

Trading direction, can be "Buy" or "Sell"

</td><td>

√

</td></tr><tr><td>

strike

</td><td>

DOUBLE

</td><td>

Strike price

</td><td>

√

</td></tr><tr><td>

dayCountConvention

</td><td>

STRING

</td><td>

The day count convention. It can be: "ActualActualISDA", "ActualActualISMA"," Actual365", "Actual360"

</td><td>

√

</td></tr><tr><td>

payoffType

</td><td>

STRING

</td><td>

Payoff type. It can be "Call" or "Put".

</td><td>

√

</td></tr><tr><td>

domesticCurve

</td><td>

STRING

</td><td>

The domestic discount curve name

</td><td>

×

</td></tr><tr><td>

foreignCurve

</td><td>

STRING

</td><td>

The foreign discount curve name

</td><td>

×

</td></tr></tbody>
</table>## Curve Field Specifications {#topic_pgr_jtq_ngc}

**IrYieldCurve**

<table id="table_n5j_nl1_4gc"><thead><tr><th>

Field Name

</th><th>

Data Type

</th><th>

Description

</th><th>

Required

</th></tr></thead><tbody><tr><td>

mktDataType

</td><td>

STRING

</td><td>

Must be "Curve"

</td><td>

√

</td></tr><tr><td>

referenceDate

</td><td>

DATE

</td><td>

Reference Date

</td><td>

√

</td></tr><tr><td>

version

</td><td>

INT

</td><td>

Version, default 0

</td><td>

×

</td></tr><tr><td>

curveType

</td><td>

STRING

</td><td>

Must be "IrYieldCurve"

</td><td>

√

</td></tr><tr><td>

dayCountConvention

</td><td>

STRING

</td><td>

The day count convention to use. It can be:

 -   "Actual360": actual/360

-   "Actual365": actual/365

-   "ActualActualISMA": actual/actual according to ISMA \(International Securities Market Association\) convention

-   "ActualActualISDA": actual/actual according to ISDA \(International Swaps and Derivatives Association\) convention.


</td><td>

√

</td></tr><tr><td>

interpMethod

</td><td>

STRING

</td><td>

Interpolation method. It can be:

 -   "Linear": linear interpolation

-   "CubicSpline": cubic spline interpolation

-   "CubicHermiteSpline": cubic Hermite interpolation


</td><td>

√

</td></tr><tr><td>

extrapMethod

</td><td>

STRING

</td><td>

Extrapolation method. It can be

 -   Flat: flat extrapolation

-   Linear: linear extrapolation


</td><td>

√

</td></tr><tr><td>

dates

</td><td>

DATE vector

</td><td>

Date of each data point

</td><td>

√

</td></tr><tr><td>

values

</td><td>

DOUBLE vector

</td><td>

Value of each data point, corresponding to the elements in dates.

</td><td>

√

</td></tr><tr><td>

curveName

</td><td>

STRING

</td><td>

Curve name

</td><td>

×

</td></tr><tr><td>

currency

</td><td>

STRING

</td><td>

Currency. It can be CNY", "USD", "EUR", "GBP", "JPY", "HKD"

</td><td>

√

</td></tr><tr><td>

compounding

</td><td>

STRING

</td><td>

The compounding interest. It can be:

 -   "Compounded": discrete compounding

-   "Simple": simple interest \(no compounding\).

-   "Continuous": continuous compounding.


</td><td>

√

</td></tr><tr><td>

settlement

</td><td>

DATE

</td><td>

Settlement date. If specified, all subsequent tenor intervals are computed starting from "settlement" rather than from "referenceDate".

</td><td>

×

</td></tr><tr><td>

frequency

</td><td>

INTEGRAL/STRING

</td><td>

The interest payment frequency. Supported values:

 -   -1 or "NoFrequency": No payment frequency

-   0 or "Once": Single lump-sum payment of principal and interest at maturity.

-   1 or "Annual": Annually

-   2 or "Semiannual": Semiannually

-   3 or "EveryFourthMonth": Every four months

-   4 or "Quarterly": Quarterly

-   6 or "BiMonthly": Every two months

-   12 or "Monthly": Monthly

-   13 or "EveryFourthWeek": Every four weeks

-   26 or "BiWeekly": Every two weeks

-   52 or "Weekly": Weekly

-   365 or "Daily": Daily

-   999 or "Other": Other frequencies


</td><td>

×

</td></tr><tr><td>

curveModel

</td><td>

STRING

</td><td>

Curve construction model; Currently, only "Bootstrap" is supported.

</td><td>

×

</td></tr><tr><td>

curveParams

</td><td>

DICT

</td><td>

Model parameters.

</td><td>

×

</td></tr></tbody>
</table>**FxVolatilitySurface**

<table id="table_rx1_b5q_ngc"><thead><tr><th>

Field Name

</th><th>

Data Type

</th><th>

Description

</th><th>

Required

</th></tr></thead><tbody><tr><td>

mktDataType

</td><td>

STRING

</td><td>

Must be "Surface"

</td><td>

√

</td></tr><tr><td>

referenceDate

</td><td>

DATE

</td><td>

Reference Date

</td><td>

√

</td></tr><tr><td>

version

</td><td>

INT

</td><td>

Version, default 0

</td><td>

×

</td></tr><tr><td>

surfaceType

</td><td>

STRING

</td><td>

Must be "FxVolatilitySurface"

</td><td>

√

</td></tr><tr><td>

smileMethod

</td><td>

STRING

</td><td>

Volatility smile method. It can be:

 -   "Linear": linear smile

-   "CubicSpline": cubic-spline smile

-   "SVI": SVI-model smile

-   "SABR": SABR-model smile


</td><td>

√

</td></tr><tr><td>

volSmiles

</td><td>

DICT\(STRING, ANY\)vector

</td><td>

Volatility smiles vector. Each element is one smile . It has the following members:

 -   **strikes**: A DOUBLE vector indicating the strike prices.

-   **vols**: A DOUBLE vector indicating the volatilities corresponding to *strikes*\(of the same length\).

-   **curveParams**: A DICT\(STRING, DOUBLE\) indicating the model parameters for the smile method; only effective when *smileMethod*is "SVI" or "SABR":

    -   smileMethod = 'SVI': Must have the keys: 'a', 'b', 'rho', 'm', 'sigma'

    -   smileMethod = 'SABR':

Must have the keys: 'alpha', 'beta', 'rho', 'nu'

-   **fwd** \(optional \): A DOUBLE scalar indicating the forward value. It is required when *smileMethod*is "SVI" or "SABR".


</td><td>

√

</td></tr><tr><td>

termDates

</td><td>

DATE vector

</td><td>

Term date corresponding to each smile in *volSmiles*.

</td><td>

√

</td></tr><tr><td>

surfaceName

</td><td>

STRING

</td><td>

Surface name

</td><td>

×

</td></tr><tr><td>

currencyPair

</td><td>

STRING

</td><td>

Foreign exchange currency pair. Available options: "EURUSD", "USDCNY", "EURCNY", "GBPCNY", "JPYCNY", "HKDCNY".

 Currency pairs may also use `.` or `/` as separators. For example, "EURUSD" can also be written as "EUR.USD" or "EUR/USD".

</td><td>

√

</td></tr></tbody>
</table>