# callableFixedRateBondDirtyPrice {#callableFixedRateBondDirtyPrice}

**Parent topic:**[Functions](../../Functions/category.md)

## Syntax {#syntax}

`callableFixedRateBondDirtyPrice(settlement, issue, maturity, redemption, coupon, riskFree, volatility, reversion, type, exerciseDates, exercisePrices, calendar, frequency, [basis=1], [convention='Following'], [method='HullWhite'], [kwargs])`

## Details {#details}

Calculate the dirty price per 100 par value for bonds with embedded options.

Bonds with options embedded in their terms grant the issuer or the holder certain rights under specific conditions. Depending on the type of option, such bonds are primarily classified as:

-   Callable bonds: Grants the issuer the right to redeem the bond at a specific price \(call price\) before maturity.
-   Putable bonds: Grants the bondholder the right to sell the bond back to the issuer at a specific price \(put price\) before maturity.

**Return value:** A scalar or vector of type DOUBLE.

## Arguments {#arguments}

**Note:** Scalar inputs will be automatically expanded to match the length of other vector inputs. All vector inputs must be of equal length. For array vector inputs, the number of rows must match the length of the other vectors.

**settlement** is a DATE scalar or vector indicating the settlement date.

**issue** is a DATE scalar or vector indicating the issuance date.

**maturity** is a DATE scalar or vector indicating the maturity date.

**redemption**is a numeric scalar or vector indicating the redemption value.

**coupon** is a numeric scalar or vector indicating the annual coupon rate. For example, 0.03 indicates a 3% annual coupon.

**riskFree** is a numeric scalar or vector indicating the risk-free interest rate.

**volatility** is a numeric scalar or vector indicating the volatility.

**reversion**is a numeric scalar or vector indicating the mean reversion.

**type** is a STRING scalar or vector that specifies the type of bonds with embedded options. The possible values are:

-   'call': callable bonds
-   'put': putable bonds

**exerciseDates** is a DATE vector or array vector that represents the exercise date.

**exercisePrices** is a numeric vector or array vector that represents the exercise price. It must have the same structure with *exerciseDates.*

**calendar** is a STRING scalar or vector indicating the trading calendar\(s\). See [Trading Calendar](../../Tutorials/trading_calendar.md) for more information.

**frequency** is an INT scalar/vector indicating the number of payments, or a STRING scalar/vector indicating payment frequency. It can be:

-   0/"Once": Bullet payment at maturity.
-   1/"Annual": Annual payments.
-   2/"Semiannual": Semi-annual payments.
-   4/"Quarterly": Quarterly payments.
-   12/"Monthly": Monthly payments.

**convention** \(optional\) is a STRING scalar or vector indicating how cash flows that fall on a non-trading day are treated. The following options are available. Defaults to 'Following'.

-   'Following': The following trading day.
-   'ModifiedFollowing': The following trading day. If that day is in a different month, the preceding trading day is adopted instead.
-   'Preceding': The preceding trading day.
-   'ModifiedPreceding': The preceding trading day. If that day is in a different month, the following trading day is adopted instead.
-   'Unadjusted': Unadjusted.
-   'HalfMonthModifiedFollowing': The following trading day. If that day crosses the mid-month \(15th\) or the end of month, the preceding trading day is adopted instead.
-   'Nearest': The nearest trading day. If both the preceding and following trading days are equally far away, default to following trading day.

**method** \(optional\) is a STRING scalar indicating the valuation method used. Currently, only "HullWhite" is supported, which means the valuation is performed using the Hull-White model.

**kwargs** \(optional\) is a dictionary specifying other parameters required by the valuation method. When *method*='HullWhite', the key-values pairs should be:

-   'timeSteps': An INT scalar or vector indicating the number of time intervals used for discretizing the Hull–White model. The default value is 50.

## Examples {#examples}

Consider an eight-year callable fixed-rate bond issued on September 16, 2024, maturing on September 15, 2032. The current date is August 16, 2026. The bond has a 2.5% annual coupon, paid quarterly, and a redemption price of 100. Starting from September 15, 2026, the issuer has the option to call the bond every three months at a price of 100. The market's risk-free rate is 4.65%, with a volatility of 20% and a mean reversion rate of 6%. The bond follows the Actual/365 day count convention, with the "Following" method for non-business day adjustments, , following the NewYork Stock Exchange \(XNYS\) calendar. Calculate the price of the bond, including accrued interest.

``` {#codeblock_zq1_tms_c2c}
issue = 2024.09.16
settlement = 2026.08.16
maturity = 2032.09.15

redemption = 100
riskFree = 0.0465
reversion = 0.06
volatility = 0.20

convention = `Following
calendar = `SSE
frequency = 3M
coupon = 0.025
basis = 3

type="call"
callDate = 2026.09.15
exerciseDates = array(DATE, 0);
for (i in 1..24) {
	exerciseDates.append!(callDate)
	callDate = temporalAdd(callDate, frequency, `Null)
}

exercisePrices = take([100.0], 24)

res = callableFixedRateBondDirtyPrice(settlement, issue, maturity, redemption, coupon, riskFree, volatility, reversion, type, exerciseDates, exercisePrices, calendar, frequency, basis, convention)
res
// output:53.603309242600587
```

