# append! {#append}

## Syntax {#syntax}

`append!(obj, newData)`

Alias: `push!`

## Arguments {#arguments}

**obj** is a local variable, and it must be a vector/tuple/matrix/table/set.

**newData** is a scalar/vector/tuple/table/set.

-   If *obj* is a vector, *newData* is a scalar, vector, or tuple whose elements are of the same type as *obj*. The result is a vector longer than *obj*.
-   If *obj* is a tuple, *newData* is a scalar, vector or tuple:
    -   If *newData* is a vector, it is appended to *obj* as one tuple element;

    -   If *newData* is a tuple, the *appendTupleAsAWhole* configuration parameter controls whether it is appended to *obj* as one tuple element \(true\) or each of its elements is appended independently \(false\).

-   If *obj* is a matrix, *newData* is a vector whose length must be a multiple of the number of rows of *obj*. The result is a matrix with the same number of rows as *obj* but with more columns.
-   If *obj* is a table, *newData* is a table with the same number of columns as *obj*. The result is a table with the same number and name of columns as *obj* but with more rows.
-   If *newData* and *obj* are of different data forms, `append!` will attempt to convert *newData* to the same data form as *obj*. If it is not possible, return an error message.

## Details {#details}

Append *newData* to *obj*. The exclamation mark \(!\) means in-place change in DolphinDB.

Note: In most cases, the column names and orders in the tables should be consistent. Please first check whether the corresponding columns in *obj* and *newData* have the same names and are arranged in the same order before executing `append!`. The function does not check the consistency of column names or align the columns if they are not arranged in the same order. It is executed as long as the corresponding columns are of the same data types.

## Examples {#examples}

```
x = 1 2 3
x.append!(4)
x
// output: [1,2,3,4]

append!(x, 5 6)
x
//output: [1,2,3,4,5,6]

x.append!(7.2)
x
//output: [1,2,3,4,5,6,7]
// converted DOUBLE 7.2 to INT 7

x.append!(`XOM)
// Error: Incompatible type. Expected: INT, Actual: STRING

x=array(int, 0, 10) // x is an empty vector
x
//output: []

x.append!(1)
x
//output: [1]

x=array(symbol, 0, 100)
append!(x, `TEST)
x
//output
["TEST"]

x=1..6$3:2
x

x = (1,"X")
y = (2,"Y")
x.append!(y)
print(x)
// when appendTupleAsAWhole = true
(1,"X",(2,"Y"))
// when appendTupleAsAWhole = false
(1,"X",2,"Y")
```

|0|1|
|---|---|
|1|4|
|2|5|
|3|6|

```
x.append!(7..12)
x
```

|0|1|2|3|
|---|---|---|---|
|1|4|7|10|
|2|5|8|11|
|3|6|9|12|

```
x=set(1 2 3 4)
x.append!(6)
x
// output: set(6,1,2,3,4)

t1=table(1 2 3 as x, 4 5 6 as y)
t2=table(1.1 2.2 3.3 as a, 4.4 5.5 6.6 as b)
t1.append!(t2)
t1
```

|x|y|
|---|---|
|1|4|
|2|5|
|3|6|
|1|4|
|2|6|
|3|7|

Use `append!` to add data to a DFS table. The following example should be executed in a DFS cluster.

```
n=1000000
t=table(rand(`IBM`MS`APPL`AMZN,n) as symbol, rand(10.0, n) as value)
db = database("dfs://rangedb_tradedata", RANGE, `A`F`M`S`ZZZZ)
Trades = db.createPartitionedTable(t, "Trades", "symbol")
```

We have created an empty table Trades with the schema of t. Next, we append the empty table Trades with data from table t.

```
Trades.append!(t)
select count(*) from Trades;
// output: 1000000
```

To append table Trades with another table:

```
n=500000
t1=table(rand(`FB`GE`MSFT,n) as symbol, rand(100.0, n) as value)
Trades.append!(t1)
select count(*) from Trades
// output: 1500000
```

