library("tidyverse")
library("magrittr")
library("ggplot2")

n <- 300000           # Number of simulated paths
s <- 1.0              # Initial underlying price
k_in <- 0.85          # Knock-in barrier
k_out <- 1.03         # Knock-out barrier
t <- 1                # Maturity (years)
days_in_year <- 252   # Number of trading days per year
sigma <- 0.13         # Annualized volatility of the underlying
r <- 0.03             # Annualized risk-free rate
lock_period <- 0      # Lock-up period (days)
up_limit <- 0.1       # Daily upward price limit
down_limit <- 0.1     # Daily downward price limit
profit_rate <- 0.01   # Issuer target profit rate for the snowball option
set.seed(42)

generate_path <- function(s, r, t, sigma, n, days_in_year, up_limit = NULL, down_limit = NULL) {
  dt <- 1 / days_in_year
  tdays <- t * days_in_year
  
  # Generate the time matrix (tdays+1 rows, n columns)
  all_t <- dt * seq(0, tdays) %>% matrix(nrow = 1)
  time_mat <- matrix(rep(all_t, each = n), nrow = tdays + 1, byrow = TRUE)
  
  # Generate the standard normal matrix
  norm_mat <- matrix(rnorm((tdays + 1) * n), nrow = tdays + 1, ncol = n)

  # Compute cumulative sums and generate the price matrix
  cum_norm <- apply(norm_mat, 2, cumsum)
  price_mat <- s * exp((r - 0.5 * sigma^2) * time_mat + sigma * sqrt(dt) * cum_norm)

  # Apply daily price limit constraints
  if (!is.null(up_limit) || !is.null(down_limit)) {
    for (i in 2:(tdays + 1)) {
      prev_price <- price_mat[i - 1, ]
      curr_price <- price_mat[i, ]
      
      upper <- if (!is.null(up_limit)) prev_price * (1 + up_limit) else Inf
      lower <- if (!is.null(down_limit)) prev_price * (1 - down_limit) else -Inf
      
      price_mat[i, ] <- pmax(lower, pmin(upper, curr_price))
    }
  }

  return(price_mat)
}

snowball_pricing <- function(s, r, t, days_in_year, sigma, n, k_in, k_out, lock_period, profit_rate,
                             up_limit = NULL, down_limit = NULL) {

  calculate_payoff <- function(s, r, t, coupon, knock_out_day, knock_in_day, days_in_year){
    # Knock-out occurs
    payoff1 <- coupon * (knock_out_day / days_in_year) * exp(-r * (knock_out_day / days_in_year))
        
    # No knock-in and no knock-out
    payoff2 <- coupon * exp(-r * t)
  
    # Knock-in occurs but no knock-out
    payoff3 <- (price_path[nrow(price_path), ] - s) * exp(-r * t)
  
    # Combined payoff
    payoff <- ifelse(knock_out_day > 0, payoff1,
                   ifelse(knock_in_day == 0, payoff2,
                          ifelse(payoff3 < 0, payoff3, 0)))
    return (mean(payoff))
  }

  tdays <- t * days_in_year
  month_day <- days_in_year %/% 12
  
  # Generate pricing paths
  price_path <- generate_path(s, r, t, sigma, n, days_in_year, up_limit, down_limit)

  #Construct observation dates for knock-out checks
  observation_out_idx <- seq(month_day, tdays, by = month_day)
  observation_out_idx <- observation_out_idx[observation_out_idx > lock_period]  
  
  # Build knock-out indicator matrix
  knock_out_matrix <- price_path[observation_out_idx, ] >= k_out 
  knock_out_days <- knock_out_matrix * observation_out_idx
  
  # Obtain the first knock-out date
  knock_out_day <- apply(knock_out_days, 2, function(x) {
    x[x <= 0] <- NA
    min(x, na.rm = TRUE)
  })
  knock_out_day[is.infinite(knock_out_day)] <- 0  # Handle infinite values

  # Determine whether knock-in occurs (price falls below barrier at any time)
  knock_in_day <- apply(price_path[2:(tdays + 1), ] < k_in, 2, any)
    
  # Count occurrences of each scenario
  knock_out_times <- sum(knock_out_day > 0)
  existence_times <- sum(knock_out_day <= 0 & !knock_in_day)
  knock_in_times <- sum(knock_out_day <= 0 & knock_in_day)
  lose_times <- sum(knock_out_day <= 0 & knock_in_day & price_path[nrow(price_path), ] < s)
  
  # Generate the initial coupon test range
  coupon_test_range <- seq(0.0, 1.0, length.out = 1000)

  # Search via for-loop iteration
  last_value <- 0
  last_coupon <- 0
  target <- 1-profit_rate
  for (i in 1:length(coupon_test_range)){
    coupon <- coupon_test_range[i]
    value <- 1 + calculate_payoff(s, r, t, coupon, knock_out_day, knock_in_day, days_in_year)   # Present Value
    if ((last_value < target) && (target < value)){
        break
    }
    last_value <- value
    last_coupon <- coupon
  } 

  # Identify the coupon interval bounds
  coupon_lower_bound <- last_coupon
  coupon_upper_bound <- coupon

  return (list(
    knock_out_times = knock_out_times,
    knock_in_times = knock_in_times,
    existence_times = existence_times,
    lose_times = lose_times,
    coupon = 0.5*(coupon_lower_bound + coupon_upper_bound)
  ))
}

# Precompile
snowball_pricing(s, r, t, days_in_year, sigma, n, k_in, k_out, lock_period, profit_rate,
                             up_limit = up_limit, down_limit = down_limit)
t0 <- Sys.time()        # Start time
res <- snowball_pricing(s, r, t, days_in_year, sigma, n, k_in, k_out, lock_period, profit_rate,
                             up_limit = up_limit, down_limit = down_limit)
t1 <- Sys.time()        # End time
print(res)
print(t1 - t0)

# Performance test
N_list = seq(10000, 500001, 10000)
record_time <- vector()
coupon_rate <- vector()
for (N in N_list){
  t0 <- Sys.time()
  res <- snowball_pricing(s, r, t, days_in_year, sigma, N, k_in, k_out, lock_period, profit_rate,
                             up_limit = up_limit, down_limit = down_limit)
  t1 <- Sys.time()
  record_time <- c(record_time, 1000*(t1 - t0))
  coupon_rate <- c(coupon_rate, res$coupon)
  print(N)
}
result = data.frame(N=N_list, time=record_time, Coupon=coupon_rate)

# Save performance test results
write.csv(result,"D:/result_seller(R).csv", row.names = FALSE)

df = read.csv("D:/result_seller(R).csv")
scale_factor <- max(df$Coupon) / max(df$time)

ggplot(df, aes(x = N)) +
  geom_line(aes(y = time, color = "Time"), size = 1) +
  geom_point(aes(y = time, color = "Time"), size = 3) +
  geom_line(aes(y = Coupon / scale_factor, color = "Coupon"), size = 1) +
  geom_point(aes(y = Coupon / scale_factor, color = "Coupon"), size = 3) +
  # Set up dual Y-axes
  scale_y_continuous(
    name = "Time(ms)",
    sec.axis = sec_axis(trans=~.*scale_factor, name="Coupon")
  ) +
  scale_color_manual(
    name = "Metrics",
    values = c("Time" = "steelblue", "Coupon" = "red")
  ) +
  theme_minimal() +
  theme(
    axis.title.y.left = element_text(color = "steelblue"),
    axis.text.y.left = element_text(color = "steelblue"),
    axis.title.y.right = element_text(color = "red"),
    axis.text.y.right = element_text(color = "red"),
    legend.position = "top",          # Place legend at the top
    panel.border = element_rect(color = "black", fill = NA, size = 1)         # Add black border
  )