import time
import tqdm
import pandas as pd
import numpy as np
from typing import Dict
import matplotlib.pyplot as plt
plt.rcParams['font.sans-serif'] = ['KaiTi']
plt.rcParams['axes.unicode_minus'] = False
np.random.seed(42)

"""
Calculate coupon rate given the issuer target profit (snowball pricing)
Assume dividend coupon = knock-out coupon
"""

def generate_path(s: float, r: float, t: float, sigma: float, n: int, days_in_year: int,
                  up_limit: float = None,  down_limit: float = None) -> np.matrix:
    """
    Generate N simulated underlying price paths based on GBM assumption
    """
    dt = 1 / days_in_year
    tdays = int(t * days_in_year)

    # Generate the time series
    all_t = dt * np.arange(tdays+1).reshape(1,-1).astype(np.float32)   # The indexing of arange in NumPy differs from DolphinDB Python parser, so we add +1 here
    time_mat = all_t.repeat(n, axis=0).transpose()

    # Generate the standard normal matrix
    norm_mat = np.random.standard_normal((tdays+1, n)).astype(np.float32)
    norm_mat = (norm_mat * (all_t>0).transpose())

    # Construct price matrix based on option Black-Scholes formula
    price_mat = s * np.exp((r-0.5*sigma*sigma) * time_mat + sigma * np.sqrt(dt)*np.cumsum(norm_mat,axis=0) )    

    # Apply daily price limit constraints (up/down limits)
    if up_limit or down_limit:
        for t in range(1, tdays + 1):
            prev_price = price_mat[t - 1, :]
            curr_price = price_mat[t, :]

            upper = prev_price * (1 + up_limit) if up_limit else np.inf
            lower = prev_price * (1 - down_limit) if down_limit else -np.inf

            price_mat[t, :] = np.clip(curr_price, lower, upper)
    return price_mat

def snowball_pricing(s: float, r: float, t: float, days_in_year: int, sigma: float, n: int,
                     k_in: float, k_out: float, lock_period: int, profit_rate: float,
                     up_limit: float = None, down_limit: float = None) -> Dict:
    """
    Compute statistical metrics of snowball payoff for simulated paths
    Note: the coupon is determined via iterative calibration
    """

    def calculate_payoff(coupon: float) -> float:
        """
        Given the coupon (assuming dividend coupon = knock-out coupon), discount to compute payoff
        """
        # Knock-out occurs
        payoff1 = coupon * (knock_out_day / days_in_year) * np.exp(-r * (knock_out_day / days_in_year))
        # No knock-in and no knock-out
        payoff2 = coupon * np.exp(-r * t)
        # Knock-in occurs but no knock-out
        payoff3 = (price_path[-1, :] - s) * np.exp(-r * t)
        payoff = np.where(knock_out_day > 0, payoff1, np.where(knock_in_day == 0, payoff2, np.where(payoff3 < 0, payoff3, 0)))
        return float(np.mean(payoff))

    tdays = t * days_in_year
    month_day = int(days_in_year/12)
    price_path = generate_path(s, r, t, sigma, n, days_in_year, up_limit, down_limit)

    # Generate observation times for knock-out checks
    observation_out_idx = np.arange(month_day, tdays+1, month_day)
    observation_out_idx = observation_out_idx[observation_out_idx>lock_period]-1

    # Determine whether knock-out occurs each month
    knock_out_matrix = price_path[observation_out_idx, :] >= k_out
    observation_out_idx = np.expand_dims(observation_out_idx, 0).repeat(price_path.shape[1], axis=0).T
    knock_out_days = knock_out_matrix * observation_out_idx
    knock_out_day = np.nanmin(np.where(knock_out_days > 0, knock_out_days, np.nan), axis=0)

    # Determine whether knock-out occurs
    knock_in_day = np.sum(price_path < k_in, axis=0)

    # Count occurrences of the three scenarios
    knock_out_times = np.sum(knock_out_day > 0)
    existence_times = np.sum(np.logical_and(np.logical_or(knock_out_day <= 0, np.isnan(knock_out_day)), knock_in_day == 0))
    knock_in_times = np.sum(np.logical_and(np.logical_or(knock_out_day <= 0, np.isnan(knock_out_day)), knock_in_day > 0))

    # Count number of losses for snowball option buyer
    lose_times = np.sum(np.logical_and(
        np.logical_and(
            np.logical_or(knock_out_day <= 0, np.isnan(knock_out_day)),
            knock_in_day > 0),
        price_path[-1, :] < s)
    )

    # Generate initial range of coupon rates
    coupon_test_range = np.linspace(0.0, 1.0, 1000)

    # Search via for-loop iteration
    last_value, last_coupon = 0, 0
    target = 1-profit_rate
    for i in range(len(coupon_test_range)):
        coupon = float(coupon_test_range[i])
        value = 1 + calculate_payoff(coupon)
        if value > target > last_value:
            break
        last_value = value
        last_coupon = coupon

    # Obtain the interval boundaries
    coupon_low = last_coupon
    coupon_high = coupon

    return {
        "knock_out_times": knock_out_times,
        "knock_in_times": knock_in_times,
        "existence_times": existence_times,
        "lose_times": lose_times,           # Number of losses for snowball option buyer
        "coupon_rate": 0.5 * (coupon_high + coupon_low),
    }


if __name__ == "__main__":
    _n = 300000               # Number of simulated paths
    _s = 1.0                  # Initial underlying price
    _k_in = 0.85 * _s         # Knock-in barrier
    _k_out = 1.03 * _s        # Knock-out barrier
    _t = 1                    # Maturity (years)
    _days_in_year = 252       # Number of trading days per year
    _sigma = 0.13             # Annualized volatility of the underlying
    _r = 0.03                 # Annualized risk-free rate
    _lock_period = 0          # Lock-up period (days)
    _profit_rate = 0.01
    _up_limit = 0.1           # Daily upward price limit for simulated paths
    _down_limit = 0.1         # Daily downward price limit for simulated paths
    snowball_pricing(_s, _r, _t, _days_in_year, _sigma, _n, _k_in, _k_out, _lock_period, _profit_rate, _up_limit, _down_limit)
    t0 = time.time()
    res_dict = snowball_pricing(_s, _r, _t, _days_in_year, _sigma, _n, _k_in, _k_out, _lock_period, _profit_rate,
                                _up_limit, _down_limit)
    t1 = time.time()
    print("runtime:",t1-t0)
    print(res_dict)

    n_list = range(10000, 500001, 10000)
    coupon_record = []
    time_record = []
    for _n in tqdm.tqdm(n_list, desc="Iterating..."):
        t0 = time.time()
        res_dict = snowball_pricing(_s, _r, _t, _days_in_year, _sigma, _n, _k_in, _k_out, _lock_period, _profit_rate,
                                _up_limit, _down_limit)
        t1 = time.time()
        coupon_record.append(res_dict["coupon_rate"])
        time_record.append(1000 * (t1 - t0))

    result = pd.DataFrame({"N": n_list,
                           "coupon_rate": coupon_record,
                           "time": time_record,
                           })
    result.to_csv(r".\Result\result_seller(Python).csv", index=False)
    result = pd.read_csv(r".\Result\result_seller(Python).csv", index_col=None)

    # Create figure and primary axes
    fig, ax1 = plt.subplots(figsize=(10, 6))

    # Primary Y-axis: plot time (ms)
    color = 'tab:blue'
    ax1.set_xlabel('N (Simulated Paths)')
    ax1.set_ylabel('Time (ms)', color=color)
    ax1.plot(result["N"], result["time"], color=color, label='Time (ms)', linewidth=2)
    ax1.tick_params(axis='y', labelcolor=color)

    # Secondary Y-axis: plot payoff
    ax2 = ax1.twinx()        # Share the same X-axis
    color = 'tab:red'
    ax2.set_ylabel('Coupon Rate', color=color)
    ax2.plot(result["N"], result["coupon_rate"], color=color, linestyle='-', label='Coupon Rate', linewidth=2)
    ax2.tick_params(axis='y', labelcolor=color)

    # Title and legend
    fig.tight_layout()
    fig.legend(loc="upper left", bbox_to_anchor=(0.1, 0.9), frameon=False)

    # Display the figure
    plt.savefig(r".\Figure\result_seller(Python).png")
    plt.show()

