import json
import time
import os
import asyncio
import queue
from typing import List
from okx_framework import OKXBaseConfig, normalize_row

class OKXLiquidationConfig(OKXBaseConfig):
    """OKX liquidation data configuration"""
    TIMEOUT = 3600  # Keep the setting
    
    # Table name and cache file
    tableName = "Cryptocurrency_liquidationST"  
    BUFFER_FILE = "./okx_liquidation_fail_buffer.jsonl"
    def __init__(self, inst_ids = None):
        super().__init__()
        self.inst_ids = inst_ids or []    
    def get_create_table_script(self) -> str:
        return '''
dbName = "dfs://CryptocurrencyDay"
tbName = "liquidation"
streamtbName = "Cryptocurrency_liquidationST"
colNames = `eventTime`collectionTime`symbolSource`symbol`side`type`timeInForce`quantity`price`avgPrice`status`LastFilledQuantity`FilledAccumulatedQuantity
colTypes = [TIMESTAMP, TIMESTAMP, SYMBOL, SYMBOL,STRING,STRING,STRING,DOUBLE,DOUBLE,DOUBLE,STRING,DOUBLE,DOUBLE]  
if(!existsDatabase(dbName)){
    //dropDatabase(dbName)
    db = database(dbName,RANGE,2010.01M+(0..20)*60)
}else{ db=database(dbName)}
if(!existsTable(dbName,tbName)){
    createPartitionedTable(db,table(1:0,colNames,colTypes),tbName,`eventTime)  
}
enableTableShareAndPersistence(table=keyedStreamTable(`symbolSource`symbol`eventTime, 10000:0, colNames, colTypes), tableName=streamtbName, cacheSize=100000, retentionMinutes=2880)
go
liquidationTb = loadTable(dbName, tbName)
subscribeTable(tableName=streamtbName, actionName="insertDB", offset=-2, handler=liquidationTb, msgAsTable=true, batchSize=10000, throttle=1, persistOffset=true)
        '''
    
    def get_subscription_args(self, inst_ids: List[str]) -> List[dict]:
        """Obtain subscriptio parameters"""
        return [{"channel": "liquidation-orders", "instType": "SWAP"}]
    
    def handle_message(self, message: str):
        """Process liquidation messages"""
        try:
            d = json.loads(message)
        except json.JSONDecodeError as e:
            print(f"[{time.strftime('%H:%M:%S')}] JSON parsing error: {e}")
            return
        
        # Check message format
        if "data" not in d or not d["data"]:
            return
        
        # Current time (UTC+8, in milliseconds)
        now_ms = int(time.time() * 1000) 
        
        # Process data
        for item in d["data"]:
            inst_id = item.get("instId")
            if not inst_id:
                continue
            
            # Filter by whitelist
            if hasattr(self, "inst_ids") and self.inst_ids:
                if inst_id not in self.inst_ids:
                    continue
            # Symbol conversion: BTC-USDT-SWAP -> BTCUSDT
            parts = inst_id.split("-")
            if len(parts) >= 2:
                symbol = parts[0] + parts[1]
            else:
                symbol = inst_id.replace("-", "")
            
            # Process details array
            for detail in item.get("details", []):
                try:
                    # Timestamp conversion
                    event_time = int(detail.get('ts', 0))
                    
                    # Construct data rows according to the original table schema
                    row = [
                        event_time,                        # eventTime
                        now_ms,                            # collectionTime
                        "OKX-Futures",                     # symbolSource
                        symbol,                            # symbol
                        detail.get('side', ''),            # side
                        None,                              # type
                        None,                              # timeInForce
                        float(detail.get('sz', 0)),        # quantity
                        float(detail.get('bkPx', 0)),      # price
                        None,                              # avgPrice
                        None,                              # status
                        None,                              # LastFilledQuantity
                        None                               # FilledAccumulatedQuantity
                    ]
                    # Join queue
                    try:
                        self.realtime_q.put(row, block=False)
                    except queue.Full:
                        # Write to local storage when the queue is full.
                        with self.file_lock, open(self.BUFFER_FILE, "a", encoding="utf-8") as f:
                            f.write(json.dumps(normalize_row(row), ensure_ascii=False) + "\n")
                            
                except (ValueError, KeyError) as e:
                    print(f"[{time.strftime('%H:%M:%S')}] Data processing error: {e}")
                    continue

def main():
    """main function"""
    # Contract list
    inst_ids = [
        "BTC-USDT-SWAP",
        "ETH-USDT-SWAP",
        "ADA-USDT-SWAP",
        "ALGO-USDT-SWAP",
        "BNB-USDT-SWAP",
        "FIL-USDT-SWAP",
        "GRT-USDT-SWAP",
        "LTC-USDT-SWAP",
        "XRP-USDT-SWAP"
    ]
    
    # Create configuration
    config = OKXLiquidationConfig(inst_ids=inst_ids)
    
    # Start data pipeline
    io_thread, okx_thread = config.start(inst_ids)
    
    # Keep the main thread live
    try:
        while True:
            time.sleep(1)
    except KeyboardInterrupt:
        print(f"\n[{time.strftime('%H:%M:%S')}] Stopping...")
        if okx_thread.ws:
            okx_thread.loop.call_soon_threadsafe(
                asyncio.create_task, okx_thread.ws.stop()
            )
        time.sleep(1.0)
        print(f"[{time.strftime('%H:%M:%S')}] Program exited")
        os._exit(0)

if __name__ == "__main__":
    main()