import time
import os
import asyncio
import json
import numpy as np

from okx_framework import OKXBaseConfig

class Depth400Config(OKXBaseConfig):

    # Table name and buffer file (must be set by subclass)
    # If the table name is modified here, the subscription table name in createDatabase.dos must also be updated
    # If the name is changed, the startup script must be updated as well
    tableName = "Cryptocurrency_raw_depth400"
    engineName = "Cryptocurrency_Okex_depth400"
    outputTableName = "Cryptocurrency_Okex_OrderBook_400"
    orderBook_action = 'subscribe_raw_orderbook'
    save_action = 'save_orderbook_to_database'
    dbName = 'dfs://CryptocurrencyOrderBook'
    tbName = 'OkexOrderbooks'
    rawtbName = 'rawOkexOrderbooks'

    BUFFER_FILE = "./okx_depth_400.jsonl"

    def __init__(self):
        super().__init__()


    def get_create_table_script(self) -> str:
        return \
        f'''
        // If the market data does not reconnect, and no full snapshot arrives, the snapshot engine will not output, so there is no need to consider state persistence. 
        // When the program restarts, if the engine does not exist, subscriptions must be created in advance. If the engine already exists, no action is needed; once full data re-enters the engine, the state should restart automatically.
        // Therefore, we only need to ensure that when the program starts, the stream table already exists and the engine has been created.
        inputName = "{Depth400Config.tableName}"
        engineName = "{Depth400Config.engineName}"
        outputName = "{Depth400Config.outputTableName}"
        saveactionName = "{Depth400Config.save_action}"
        orderbook_action = "{Depth400Config.orderBook_action}"
        dbName = "{Depth400Config.dbName}"
        tbName = "{Depth400Config.tbName}"
        rawtbName = "{Depth400Config.rawtbName}"

        // Create the stream table for receiving market data, not persisted
        if (not existsStreamTable(inputName)){{
            colNames=['isIncremental', 'symbol', 'askPrice', 'askVolume', 'askNum', 'bidPrice', 'bidVolume', 'bidNum', 'checksum', 'prevSeqId', 'seqId', 'updateTime']
            colTypes=[BOOL, SYMBOL, DOUBLE[], DOUBLE[], INT[], DOUBLE[], DOUBLE[], INT[], LONG, LONG, LONG, TIMESTAMP]
            
            enableTableShareAndPersistence(keyedStreamTable(`symbol`updateTime, 1:0, colNames, colTypes), tbName, cacheSize=100000, preCache=2000)
            // enableTableShareAndCachePurge(streamTable(1:0, colNames, colTypes), inputName, cacheSize=100000)

            // If the stream table does not exist, rebuild it. The engine definitely does not exist in this case. For safety, try dropping it once.
            try{{unsubscribeTable(, inputName, orderbook_action)}}catch(ex){{}}
            try{{dropStreamEngine(engineName)}}catch(ex){{}}
        }}

        // Create the stream table used to receive snapshots synthesized by the engine. Since the data will later be written to the database, persistence is not required.
        if (not existsStreamTable(outputName)){{
            colNames=['isIncremental', 'symbol', 'askPrice', 'askVolume', 'askNum', 'bidPrice', 'bidVolume', 'bidNum', 'checksum', 'prevSeqId', 'seqId', 'updateTime'];
            colTypes=[BOOL, SYMBOL, DOUBLE[],DOUBLE[], INT[],DOUBLE[], DOUBLE[], INT[], LONG, LONG, LONG, TIMESTAMP];
            
            enableTableShareAndPersistence(keyedStreamTable(`symbol`updateTime, 1:0, colNames, colTypes), outputName, cacheSize=100000, preCache=2000)
            // enableTableShareAndCachePurge(streamTable(1:0, colNames, colTypes), outputName, cacheSize=100000)

            // This stream table is used to receive order books synthesized by the stream engine
        }}


        // Create the engine if it does not exist
        try{{engine = getStreamEngine(engineName)}}catch(ex){{engine = NULL}}
        if (isNull(engine)){{
            inputTarget = ["symbol", "eventTime", "isIncremental", "bidPrice", "bidQty", "askPrice", "askQty", "prevUpdateId", "updateId"];
            inputSource = ["symbol", "updateTime", 'isIncremental', 'bidPrice', 'bidVolume', 'askPrice', 'askVolume', 'prevSeqId', 'seqId'];
            inputColMap = dict(inputTarget, inputSource)

            def errorHandler(instrument, code) {{
                if (code == 1) {{
                    writeLog("handle historical msg...")
                }} else if (code == 2) {{
                    writeLog("handle unordered msg...")
                }} else if (code == 3) {{
                    writeLog("handle timeout...")
                }} else if (code == 4) {{
                    writeLog("handle crossed price...")
                }} else {{
                    writeLog("unknown error!")
                }}
            }}

            // One copy goes to the stream table, another is written to the database
            def msgHandler(outputName, dbName, tbName, msg){{
                objByName(outputName).append!(msg)
                loadTable(dbName, tbName).append!(msg)
            }}

            engine = createCryptoOrderBookEngine(
                        name=engineName, 
                        dummyTable=objByName(inputName), 
                        inputColMap=inputColMap, 
                        //outputTable=objByName(outputName),
                        outputHandler=msgHandler{{outputName,dbName,tbName}},
                        depth=400, updateRule="general", errorHandler=errorHandler, cachingInterval=5000, 
                        timeout=6000, msgAsTable=true, cachedDepth=400, snapshotDir=getHomeDir()+"/okex_depth400_egine", snapshotIntervalInMsgCount=10000)
            try{{unsubscribeTable(, inputName, orderbook_action)}}catch(ex){{}}
        }}

        // If the subscription already exists, there is no need to subscribe again
        if (not existsSubscriptionTopic(tableName=inputName, actionName=orderbook_action)){{
            def handler_raw_data(engineName,dbName,tbName, msg){{
                // Write into the synthesis engine
                getStreamEngine(engineName).append!(msg)

                // Write to database
                loadTable(dbName, tbName).append!(msg)
            }}
            subscribeTable(,inputName, orderbook_action, -2, handler_raw_data{{engineName, dbName, rawtbName}}, msgAsTable=true, throttle=0.1, persistOffset=true)
        }}
        
        '''

    def get_subscription_args(self, inst_ids) -> list:
        return [{"channel": "books", "instId": i} for i in inst_ids]
        # return [{"channel": "books-l2-tbt", "instId": i} for i in inst_ids]
    
    def handle_message(self, message):
        responce = json.loads(message)
        # print(responce)
        if "data" not in responce or not responce["data"]:
            return
        
        data: dict = responce["data"][0]
        asks = np.array(data["asks"], dtype=np.float64, ndmin=2)
        asks_PQO = ([], [], []) if asks.shape[1]==0 else (asks[:, 0].tolist(), asks[:, 1].tolist(), asks[:, 3].astype(np.int32).tolist())
        bids = np.array(data["bids"], dtype=np.float64, ndmin=2)
        bids_PQO = ([], [], []) if bids.shape[1]==0 else (bids[:, 0].tolist(), bids[:, 1].tolist(), bids[:, 3].astype(np.int32).tolist())
        row = [
            False if responce["action"]=='snapshot' else True,
            responce["arg"]["instId"],     # ? symbol
            asks_PQO[0], # ? askPrice
            asks_PQO[1], # ? askQty
            asks_PQO[2], # ? askOrders
            bids_PQO[0], # ? bidPrice
            bids_PQO[1], # ? bidQty
            bids_PQO[2], # ? bidOrders
            int(data["checksum"]),
            int(data["prevSeqId"]), # ? prevUpdateId
            int(data["seqId"]), # ? updateId
            int(data["ts"])+8*60*60*1000,    # ? eventTime: must use UTC+8. DolphinDB reads as UTC+8, so add 8*60*60*1000 ms here
        ]

        try:
            self.realtime_q.put(row, block=False)
        except:
            with self.file_lock, open(self.BUFFER_FILE, "a", encoding="utf-8") as f:
                f.write(json.dumps(row) + "\n")

if __name__ == "__main__":

    inst_ids = [
        "BTC-USDT-SWAP",
        "ETH-USDT-SWAP",
        "ADA-USDT-SWAP",
        "ALGO-USDT-SWAP",
        "BNB-USDT-SWAP",
        "FIL-USDT-SWAP",
        "GRT-USDT-SWAP",
        "LTC-USDT-SWAP",
        "XRP-USDT-SWAP"
    ]

    config = Depth400Config()

    # Start the data pipeline
    io_thread, okx_thread = config.start(inst_ids)
    
    # Keep the main thread alive
    try:
        while True:
            time.sleep(1)
    except KeyboardInterrupt:
        print(f"\n[{time.strftime('%H:%M:%S')}] Stopping...")
        if okx_thread.ws:
            okx_thread.loop.call_soon_threadsafe(
                asyncio.create_task, okx_thread.ws.stop()
            )
        time.sleep(1.0)
        print(f"[{time.strftime('%H:%M:%S')}] Program exited")
        os._exit(0)
