import json, time, os, asyncio
import numpy as np
from okx_framework import OKXBaseConfig

class DepthConfig(OKXBaseConfig):
    """depth data configuration """
    
    # 1. Modify table name and cache file
    tableName = "Cryptocurrency_depthST"
    BUFFER_FILE = "./OKX_depth_fail_buffer.jsonl"
    
    # 2. Modify table creation script
    def get_create_table_script(self) -> str:
        return '''
//Tick - depth 
dbName = "dfs://CryptocurrencyTick"
tbName = "depth"
streamtbName = "Cryptocurrency_depthST"
colNames = `eventTime`collectionTime`symbolSource`symbol`bidPrice`bidQty`bidOrders`askPrice`askQty`askOrders`firstId`lastId`prevLastId
colTypes = [TIMESTAMP, TIMESTAMP, SYMBOL, SYMBOL,DOUBLE[], DOUBLE[], DOUBLE[], DOUBLE[],DOUBLE[], DOUBLE[],LONG,LONG,LONG] 
if(!existsDatabase(dbName)){
    dbDate = database("", VALUE, 2012.01.01..2012.01.30)
    dbSym = database("", HASH, [SYMBOL, 2])
    db = database(dbName, COMPO, [dbDate,dbSym], engine='TSDB')    
}else{db=database(dbName)}
if(!existsTable(dbName,tbName)){
    createPartitionedTable(db,table(1:0,colNames,colTypes),tbName,`eventTime`symbol,sortColumns=`symbolSource`symbol`eventTime,keepDuplicates=LAST)  
}
enableTableShareAndPersistence(table=keyedStreamTable(`symbolSource`symbol`eventTime, 10000:0, colNames, colTypes), tableName=streamtbName, cacheSize=100000, retentionMinutes=2880)
go
depthTb = loadTable(dbName, tbName)
subscribeTable(tableName=streamtbName, actionName="insertDB", offset=-2, handler=depthTb, msgAsTable=true, batchSize=10000, throttle=1, persistOffset=true)
'''

    # 3. Modify subscription parameters
    def get_subscription_args(self, inst_ids: list) -> list:
        return [{"channel": "books5", "instId": s} for s in inst_ids]
    
    # 4. Modify message handling
    def handle_message(self, message: str):
        d = json.loads(message)
        if "data" not in d or not d["data"]:
            return
        
        j = d["data"][0]
        bid_arr = np.array(j["bids"], dtype=float)
        ask_arr = np.array(j["asks"], dtype=float)
        
        row = [
            int(j["ts"]), 
            int(time.time() * 1000),
            "OKX-Futures", 
            ''.join(j["instId"].split('-')[:-1]),
            bid_arr[:, 0].tolist(), 
            bid_arr[:, 1].tolist(), 
            bid_arr[:, 3].tolist(),
            ask_arr[:, 0].tolist(), 
            ask_arr[:, 1].tolist(), 
            ask_arr[:, 3].tolist(),
            None, None, None
        ]
        
        try:
            self.realtime_q.put(row, block=False)
        except:
            with self.file_lock, open(self.BUFFER_FILE, "a", encoding="utf-8") as f:
                f.write(json.dumps(row) + "\n")

if __name__ == "__main__":
    # Select the type of data to run
    
    # Contract list
    inst_ids = [
        "BTC-USDT-SWAP",
        "ETH-USDT-SWAP",
        "ADA-USDT-SWAP",
        "ALGO-USDT-SWAP",
        "BNB-USDT-SWAP",
        "FIL-USDT-SWAP",
        "GRT-USDT-SWAP",
        "LTC-USDT-SWAP",
        "XRP-USDT-SWAP"
    ]
    
    # Create configuration based on type

    config = DepthConfig()
    
    # Start data pipeline
    io_thread, okx_thread = config.start(inst_ids)
    
    # Keep the main thread live
    try:
        while True:
            time.sleep(1)
    except KeyboardInterrupt:
        print(f"\n[{time.strftime('%H:%M:%S')}] Stopping...")
        if okx_thread.ws:
            okx_thread.loop.call_soon_threadsafe(
                asyncio.create_task, okx_thread.ws.stop()
            )
        time.sleep(1.0)
        print(f"[{time.strftime('%H:%M:%S')}] Program exited")
        os._exit(0)