import json, time, os, asyncio
import numpy as np
from okx_framework import OKXBaseConfig

class aggTradeConfig(OKXBaseConfig):
    """Depth data configuration """
    
    # 1. Modify table name and buffer file
    tableName = "Cryptocurrency_aggTradeST"
    BUFFER_FILE = "./OKX_aggTrade_fail_buffer.jsonl"
    
    # 2. Modify create table script
    def get_create_table_script(self) -> str:
        return '''
dbName = "dfs://CryptocurrencyTick"
tbName = "trade"
streamtbName = "Cryptocurrency_aggTradeST"
colNames = `eventTime`collectionTime`symbolSource`symbol`price`quantity`count`side`tradeId`firstId`lastId
colTypes = [TIMESTAMP, TIMESTAMP, SYMBOL, SYMBOL,DOUBLE,DOUBLE,INT,STRING,LONG,LONG,LONG] 
if(!existsDatabase(dbName)){
    dbDate = database("", VALUE, 2012.01.01..2012.01.30)
    dbSym = database("", HASH, [SYMBOL, 2])
    db = database(dbName, COMPO, [dbDate,dbSym], engine='TSDB')    
}else{db=database(dbName)}
if(!existsTable(dbName,tbName)){ 
    createPartitionedTable(db,table(1:0,colNames,colTypes),tbName,`eventTime`symbol,sortColumns=`symbolSource`symbol`eventTime)  
}
enableTableShareAndPersistence(table=keyedStreamTable(`symbolSource`symbol`lastId, 10000:0, colNames, colTypes), tableName=streamtbName, cacheSize=100000, retentionMinutes=2880)
go

// Subscribe to the stream table and insert data into the DFS table
tradeTb = loadTable(dbName, tbName)
subscribeTable(tableName=streamtbName, actionName="insertDB", offset=-2, handler=tradeTb, msgAsTable=true, batchSize=10000, throttle=1, persistOffset=true)
'''

    # 3. Modify subscription arguments
    def get_subscription_args(self, inst_ids: list) -> list:
        return [{"channel": "trades", "instId": s} for s in inst_ids]
    
    # 4. Modify message handling
    def handle_message(self, message: str):
        d = json.loads(message)
        if "data" not in d or not d["data"]:
            return

        j = d["data"][0]

        row = [
            int(j["ts"]), int(time.time() * 1000),
            "OKX-Futures", ''.join(j["instId"].split('-')[:-1]),
            float(j["px"]),float(j["sz"]),int(j["count"]),j["side"],None,None,int(j["tradeId"])
        ]
        
        try:
            self.realtime_q.put(row, block=False)
        except:
            with self.file_lock, open(self.BUFFER_FILE, "a", encoding="utf-8") as f:
                f.write(json.dumps(row) + "\n")

if __name__ == "__main__":
    # Choose the type of data to run 
    
    # List of contracts
    inst_ids = [
        "BTC-USDT-SWAP",
        "ETH-USDT-SWAP",
        "ADA-USDT-SWAP",
        "ALGO-USDT-SWAP",
        "BNB-USDT-SWAP",
        "FIL-USDT-SWAP",
        "GRT-USDT-SWAP",
        "LTC-USDT-SWAP",
        "XRP-USDT-SWAP"
    ]
    
    # Create configuration based on type
    
    config = aggTradeConfig()
    
    # Start data pipeline
    io_thread, okx_thread = config.start(inst_ids)
    
    # Keep main thread alive
    try:
        while True:
            time.sleep(1)
    except KeyboardInterrupt:
        print(f"\n[{time.strftime('%H:%M:%S')}] Stopping...")
        if okx_thread.ws:
            okx_thread.loop.call_soon_threadsafe(
                asyncio.create_task, okx_thread.ws.stop()
            )
        time.sleep(1.0)
        print(f"[{time.strftime('%H:%M:%S')}] Program exited")
        os._exit(0)
