import json, time, os, asyncio
import numpy as np
from okx_framework import OKXBaseConfig

class tickTradeConfig(OKXBaseConfig):
    """Tick trade configuration"""
    # 1. Modify table name and cache file
    tableName = "Cryptocurrency_tickTradeST"
    BUFFER_FILE = "./OKX_tickTrade_fail_buffer.jsonl"
    OKX_WS_URL = "wss://ws.okx.com:8443/ws/v5/business"
    # 2. Modify table creation script
    def get_create_table_script(self) -> str:
        return '''
//Tick - tickTrade
dbName = "dfs://CryptocurrencyTick"
tbName = "tickTrade"
streamtbName = "Cryptocurrency_tickTradeST"
colNames = `eventTime`collectionTime`symbolSource`symbol`tradeId`price`quantity`quoteQty`side
colTypes = [TIMESTAMP, TIMESTAMP, SYMBOL, SYMBOL,LONG,DOUBLE,DOUBLE,DOUBLE,STRING] 
if(!existsDatabase(dbName)){
    dbDate = database("", VALUE, 2012.01.01..2012.01.30)
    dbSym = database("", HASH, [SYMBOL, 2])
    db = database(dbName, COMPO, [dbDate,dbSym], engine='TSDB')    
}else{db=database(dbName)}
if(!existsTable(dbName,tbName)){ 
    createPartitionedTable(db,table(1:0,colNames,colTypes),tbName,`eventTime`symbol,sortColumns=`symbolSource`symbol`eventTime)  
}
enableTableShareAndPersistence(table=keyedStreamTable(`symbolSource`symbol`tradeId, 10000:0, colNames, colTypes), tableName=streamtbName, cacheSize=100000, retentionMinutes=2880)
go

// Subscribe to the stream table and insert data into the DFS table
tradeTb = loadTable(dbName, tbName)
subscribeTable(tableName=streamtbName, actionName="insertDB", offset=-2, handler=tradeTb, msgAsTable=true, batchSize=10000, throttle=1, persistOffset=true)
'''

    # 3. Modify subscription parameters
    def get_subscription_args(self, inst_ids: list) -> list:
        return [{"channel": "trades-all", "instId": s} for s in inst_ids]
    
    # 4. Modify message processing
    def handle_message(self, message: str):
        d = json.loads(message)
        if "data" not in d or not d["data"]:
            return

        j = d["data"][0]

        row = [
            int(j["ts"]), int(time.time() * 1000),
            "OKX-Spot", ''.join(j["instId"].split('-')),
            int(j["tradeId"]),float(j["px"]),float(j["sz"]),None,j["side"]
        ]
        
        try:
            self.realtime_q.put(row, block=False)
        except:
            with self.file_lock, open(self.BUFFER_FILE, "a", encoding="utf-8") as f:
                f.write(json.dumps(row) + "\n")

if __name__ == "__main__":
    # Select the type of data to run
    
    # Contract list
    inst_ids = [
        "BTC-USDT",
        "ETH-USDT",
        "ADA-USDT",
        "ALGO-USDT",
        "BNB-USDT",
        "FIL-USDT",
        "GRT-USDT",
        "LTC-USDT",
        "XRP-USDT"
    ]
    
    # Create configuration based on type

    config = tickTradeConfig()
    
    # Start data pipeline
    io_thread, okx_thread = config.start(inst_ids)
    
    # Keep the main thread live
    try:
        while True:
            time.sleep(1)
    except KeyboardInterrupt:
        print(f"\n[{time.strftime('%H:%M:%S')}] Stopping...")
        if okx_thread.ws:
            okx_thread.loop.call_soon_threadsafe(
                asyncio.create_task, okx_thread.ws.stop()
            )
        time.sleep(1.0)
        print(f"[{time.strftime('%H:%M:%S')}] Program exited")
        os._exit(0)