from Binance_framework import BinanceBaseConfig, IOThread, normalize_row
from binance.websocket.um_futures.websocket_client import UMFuturesWebsocketClient
import time, json, numpy as np

class BinanceLiquidationConfig(BinanceBaseConfig):
    """Binance liquidation data ingestion config"""

    tableName = "Cryptocurrency_liquidationST"
    BUFFER_FILE = "./Binance_liquidation_fail_buffer.jsonl"
    TIMEOUT = 86400
    symbols = ["btcusdt","ethusdt","adausdt","algousdt",
               "bnbusdt","fetusdt","grtusdt","ltcusdt","xrpusdt"]
    
    def get_create_table_script(self) -> str:
        return '''
//Low-frequency - liquidation
dbName = "dfs://CryptocurrencyDay"
tbName = "liquidation"
streamtbName = "Cryptocurrency_liquidationST"
colNames = `eventTime`collectionTime`symbolSource`symbol`side`type`timeInForce`quantity`price`avgPrice`status`LastFilledQuantity`FilledAccumulatedQuantity
colTypes = [TIMESTAMP, TIMESTAMP, SYMBOL, SYMBOL,STRING,STRING,STRING,DOUBLE,DOUBLE,DOUBLE,STRING,DOUBLE,DOUBLE]  
if(!existsDatabase(dbName)){
    //dropDatabase(dbName)
    db = database(dbName,RANGE,2010.01M+(0..20)*60)
}else{ db=database(dbName)}
if(!existsTable(dbName,tbName)){
    createPartitionedTable(db,table(1:0,colNames,colTypes),tbName,`eventTime)  
}
enableTableShareAndPersistence(table=keyedStreamTable(`symbolSource`symbol`eventTime, 10000:0, colNames, colTypes), tableName=streamtbName, cacheSize=100000, retentionMinutes=2880)
go
liquidationTb = loadTable(dbName, tbName)
subscribeTable(tableName=streamtbName, actionName="insertDB", offset=-2, handler=liquidationTb, msgAsTable=true, batchSize=10000, throttle=1, persistOffset=true)
        '''
    
    def create_message_handler(self):
        def message_handler(_, message):
            self.last_received_time = time.time()
            j = json.loads(message)
            if j.get("e") != "forceOrder":
                return           
            k = j['o']
            cols = [
                j['E'] ,
                int(time.time()*1000),
                "Binance-Futures",
                k['s'],k['S'].lower(),k['o'],k['f'],
                float(k['q']),float(k['p']),float(k['ap']),
                k['X'],float(k['l']),float(k['z'])
            ]
            
            try:
                self.realtime_q.put(cols, block=False)
            except:
                # Fallback handling
                with self.file_lock, open(self.BUFFER_FILE, "a", encoding="utf-8") as f:
                    f.write(json.dumps(self.normalize_row(cols), ensure_ascii=False) + "\n")
        
        return message_handler
    
    def start_client_and_subscribe(self):
        client = UMFuturesWebsocketClient(
            on_message=self.create_message_handler(),
            proxies={'http': self.proxy_address, 'https': self.proxy_address}
        )
        
        for s in self.symbols:
            client.liquidation_order(symbol=s)
            time.sleep(0.2)
        
        return client


if __name__ == "__main__":
    config = BinanceLiquidationConfig()
    client = config.start_all()
    
    # Keep the main thread running
    try:
        while True:
            time.sleep(1)
    except KeyboardInterrupt:
        config.quick_exit()