from Binance_framework import BinanceBaseConfig, IOThread, normalize_row
from binance.websocket.um_futures.websocket_client import UMFuturesWebsocketClient
import time, json, numpy as np

class BinanceDepthConfig(BinanceBaseConfig):
    """Binance Depth data ingestion config"""
    tableName = "Cryptocurrency_depthST"
    BUFFER_FILE = "./Binance_depth_fail_buffer.jsonl"
    
    symbols = ["btcusdt","ethusdt","adausdt","algousdt",
               "bnbusdt","fetusdt","grtusdt","ltcusdt","xrpusdt"]
    
    def get_create_table_script(self) -> str:
        return '''
//Tick - depth 
dbName = "dfs://CryptocurrencyTick"
tbName = "depth"
streamtbName = "Cryptocurrency_depthST"
colNames = `eventTime`collectionTime`symbolSource`symbol`bidPrice`bidQty`bidOrders`askPrice`askQty`askOrders`firstId`lastId`prevLastId
colTypes = [TIMESTAMP, TIMESTAMP, SYMBOL, SYMBOL,DOUBLE[], DOUBLE[], DOUBLE[], DOUBLE[],DOUBLE[], DOUBLE[],LONG,LONG,LONG] 
if(!existsDatabase(dbName)){
    dbDate = database("", VALUE, 2012.01.01..2012.01.30)
    dbSym = database("", HASH, [SYMBOL, 2])
    db = database(dbName, COMPO, [dbDate,dbSym], engine='TSDB')    
}else{db=database(dbName)}
if(!existsTable(dbName,tbName)){
    createPartitionedTable(db,table(1:0,colNames,colTypes),tbName,`eventTime`symbol,sortColumns=`symbolSource`symbol`eventTime)  
}
enableTableShareAndPersistence(table=keyedStreamTable(`symbolSource`symbol`eventTime, 10000:0, colNames, colTypes), tableName=streamtbName, cacheSize=100000, retentionMinutes=2880)
go
depthTb = loadTable(dbName, tbName)
subscribeTable(tableName=streamtbName, actionName="insertDB", offset=-2, handler=depthTb, msgAsTable=true, batchSize=10000, throttle=1, persistOffset=true)
        '''
    
    def create_message_handler(self):
        def message_handler(_, message):
            self.last_received_time = time.time()
            j = json.loads(message)
            if j.get("e") != "depthUpdate":
                return
            
            try:
                bid_arr = np.array(j['b'], dtype=float)
                ask_arr = np.array(j['a'], dtype=float)
            except Exception:
                return
            
            cols = [
                j['E'],
                int(time.time()*1000),
                "Binance-Futures",
                j['s'],
                bid_arr[:,0].tolist(), bid_arr[:,1].tolist(), [],
                ask_arr[:,0].tolist(), ask_arr[:,1].tolist(), [],
                j['U'],j['u'],j['pu']
            ]
            
            try:
                self.realtime_q.put(cols, block=False)
            except:
                # Fallback handling
                with self.file_lock, open(self.BUFFER_FILE, "a", encoding="utf-8") as f:
                    f.write(json.dumps(self.normalize_row(cols), ensure_ascii=False) + "\n")
        
        return message_handler
    
    def start_client_and_subscribe(self):
        client = UMFuturesWebsocketClient(
            on_message=self.create_message_handler(),
            proxies={'http': self.proxy_address, 'https': self.proxy_address}
        )
        
        for s in self.symbols:
            client.partial_book_depth(symbol=s, level=20, speed=100)
            time.sleep(0.2)
        
        return client


if __name__ == "__main__":
    config = BinanceDepthConfig()
    client = config.start_all()
    
    # Keep the main thread running
    try:
        while True:
            time.sleep(1)
    except KeyboardInterrupt:
        config.quick_exit()